// Copyright 2023 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package chat provides access to the Google Chat API.
//
// For product documentation, see: https://developers.google.com/hangouts/chat
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/chat/v1"
//	...
//	ctx := context.Background()
//	chatService, err := chat.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for authentication.
//
// For information on how to create and obtain Application Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate. To restrict scopes, use option.WithScopes:
//
//	chatService, err := chat.NewService(ctx, option.WithScopes(chat.ChatSpacesReadonlyScope))
//
// To use an API key for authentication (note: some APIs do not support API keys), use option.WithAPIKey:
//
//	chatService, err := chat.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth flow), use option.WithTokenSource:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	chatService, err := chat.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See https://godoc.org/google.golang.org/api/option/ for details on options.
package chat // import "google.golang.org/api/chat/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint

const apiId = "chat:v1"
const apiName = "chat"
const apiVersion = "v1"
const basePath = "https://chat.googleapis.com/"
const mtlsBasePath = "https://chat.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Private Service: https://www.googleapis.com/auth/chat.bot
	ChatBotScope = "https://www.googleapis.com/auth/chat.bot"

	// View, add, and remove members from conversations in Google Chat
	ChatMembershipsScope = "https://www.googleapis.com/auth/chat.memberships"

	// View members in Google Chat conversations.
	ChatMembershipsReadonlyScope = "https://www.googleapis.com/auth/chat.memberships.readonly"

	// View, compose, send, update, and delete messages, and add, view, and
	// delete reactions to messages.
	ChatMessagesScope = "https://www.googleapis.com/auth/chat.messages"

	// Compose and send messages in Google Chat
	ChatMessagesCreateScope = "https://www.googleapis.com/auth/chat.messages.create"

	// View messages and reactions in Google Chat
	ChatMessagesReadonlyScope = "https://www.googleapis.com/auth/chat.messages.readonly"

	// Create conversations and spaces and view or update metadata
	// (including history settings) in Google Chat
	ChatSpacesScope = "https://www.googleapis.com/auth/chat.spaces"

	// View chat and spaces in Google Chat
	ChatSpacesReadonlyScope = "https://www.googleapis.com/auth/chat.spaces.readonly"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/chat.bot",
		"https://www.googleapis.com/auth/chat.memberships",
		"https://www.googleapis.com/auth/chat.memberships.readonly",
		"https://www.googleapis.com/auth/chat.messages",
		"https://www.googleapis.com/auth/chat.messages.create",
		"https://www.googleapis.com/auth/chat.messages.readonly",
		"https://www.googleapis.com/auth/chat.spaces",
		"https://www.googleapis.com/auth/chat.spaces.readonly",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s, err := New(client)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	s := &Service{client: client, BasePath: basePath}
	s.Media = NewMediaService(s)
	s.Spaces = NewSpacesService(s)
	return s, nil
}

type Service struct {
	client    *http.Client
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Media *MediaService

	Spaces *SpacesService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewMediaService(s *Service) *MediaService {
	rs := &MediaService{s: s}
	return rs
}

type MediaService struct {
	s *Service
}

func NewSpacesService(s *Service) *SpacesService {
	rs := &SpacesService{s: s}
	rs.Members = NewSpacesMembersService(s)
	rs.Messages = NewSpacesMessagesService(s)
	return rs
}

type SpacesService struct {
	s *Service

	Members *SpacesMembersService

	Messages *SpacesMessagesService
}

func NewSpacesMembersService(s *Service) *SpacesMembersService {
	rs := &SpacesMembersService{s: s}
	return rs
}

type SpacesMembersService struct {
	s *Service
}

func NewSpacesMessagesService(s *Service) *SpacesMessagesService {
	rs := &SpacesMessagesService{s: s}
	rs.Attachments = NewSpacesMessagesAttachmentsService(s)
	return rs
}

type SpacesMessagesService struct {
	s *Service

	Attachments *SpacesMessagesAttachmentsService
}

func NewSpacesMessagesAttachmentsService(s *Service) *SpacesMessagesAttachmentsService {
	rs := &SpacesMessagesAttachmentsService{s: s}
	return rs
}

type SpacesMessagesAttachmentsService struct {
	s *Service
}

// ActionParameter: List of string parameters to supply when the action
// method is invoked. For example, consider three snooze buttons: snooze
// now, snooze 1 day, snooze next week. You might use action method =
// snooze(), passing the snooze type and snooze time in the list of
// string parameters.
type ActionParameter struct {
	// Key: The name of the parameter for the action script.
	Key string `json:"key,omitempty"`

	// Value: The value of the parameter.
	Value string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Key") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Key") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ActionParameter) MarshalJSON() ([]byte, error) {
	type NoMethod ActionParameter
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// ActionResponse: Parameters that a Chat app can use to configure how
// its response is posted.
type ActionResponse struct {
	// DialogAction: Input only. A response to an event related to a dialog
	// (https://developers.google.com/chat/how-tos/dialogs). Must be
	// accompanied by `ResponseType.Dialog`.
	DialogAction *DialogAction `json:"dialogAction,omitempty"`

	// Type: Input only. The type of Chat app response.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default type; will be handled as NEW_MESSAGE.
	//   "NEW_MESSAGE" - Post as a new message in the topic.
	//   "UPDATE_MESSAGE" - Update the Chat app's message. This is only
	// permitted on a CARD_CLICKED event where the message sender type is
	// BOT.
	//   "UPDATE_USER_MESSAGE_CARDS" - Update the cards on a user's message.
	// This is only permitted as a response to a MESSAGE event with a
	// matched url, or a CARD_CLICKED event where the message sender type is
	// HUMAN. Text will be ignored.
	//   "REQUEST_CONFIG" - Privately ask the user for additional auth or
	// config.
	//   "DIALOG" - Presents a
	// [dialog](https://developers.google.com/chat/how-tos/dialogs).
	Type string `json:"type,omitempty"`

	// Url: Input only. URL for users to auth or config. (Only for
	// REQUEST_CONFIG response types.)
	Url string `json:"url,omitempty"`

	// ForceSendFields is a list of field names (e.g. "DialogAction") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DialogAction") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ActionResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ActionResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// ActionStatus: Represents the status for a request to either invoke or
// submit a dialog (https://developers.google.com/chat/how-tos/dialogs).
type ActionStatus struct {
	// StatusCode: The status code.
	//
	// Possible values:
	//   "OK" - Not an error; returned on success. HTTP Mapping: 200 OK
	//   "CANCELLED" - The operation was cancelled, typically by the caller.
	// HTTP Mapping: 499 Client Closed Request
	//   "UNKNOWN" - Unknown error. For example, this error may be returned
	// when a `Status` value received from another address space belongs to
	// an error space that is not known in this address space. Also errors
	// raised by APIs that do not return enough error information may be
	// converted to this error. HTTP Mapping: 500 Internal Server Error
	//   "INVALID_ARGUMENT" - The client specified an invalid argument. Note
	// that this differs from `FAILED_PRECONDITION`. `INVALID_ARGUMENT`
	// indicates arguments that are problematic regardless of the state of
	// the system (e.g., a malformed file name). HTTP Mapping: 400 Bad
	// Request
	//   "DEADLINE_EXCEEDED" - The deadline expired before the operation
	// could complete. For operations that change the state of the system,
	// this error may be returned even if the operation has completed
	// successfully. For example, a successful response from a server could
	// have been delayed long enough for the deadline to expire. HTTP
	// Mapping: 504 Gateway Timeout
	//   "NOT_FOUND" - Some requested entity (e.g., file or directory) was
	// not found. Note to server developers: if a request is denied for an
	// entire class of users, such as gradual feature rollout or
	// undocumented allowlist, `NOT_FOUND` may be used. If a request is
	// denied for some users within a class of users, such as user-based
	// access control, `PERMISSION_DENIED` must be used. HTTP Mapping: 404
	// Not Found
	//   "ALREADY_EXISTS" - The entity that a client attempted to create
	// (e.g., file or directory) already exists. HTTP Mapping: 409 Conflict
	//   "PERMISSION_DENIED" - The caller does not have permission to
	// execute the specified operation. `PERMISSION_DENIED` must not be used
	// for rejections caused by exhausting some resource (use
	// `RESOURCE_EXHAUSTED` instead for those errors). `PERMISSION_DENIED`
	// must not be used if the caller can not be identified (use
	// `UNAUTHENTICATED` instead for those errors). This error code does not
	// imply the request is valid or the requested entity exists or
	// satisfies other pre-conditions. HTTP Mapping: 403 Forbidden
	//   "UNAUTHENTICATED" - The request does not have valid authentication
	// credentials for the operation. HTTP Mapping: 401 Unauthorized
	//   "RESOURCE_EXHAUSTED" - Some resource has been exhausted, perhaps a
	// per-user quota, or perhaps the entire file system is out of space.
	// HTTP Mapping: 429 Too Many Requests
	//   "FAILED_PRECONDITION" - The operation was rejected because the
	// system is not in a state required for the operation's execution. For
	// example, the directory to be deleted is non-empty, an rmdir operation
	// is applied to a non-directory, etc. Service implementors can use the
	// following guidelines to decide between `FAILED_PRECONDITION`,
	// `ABORTED`, and `UNAVAILABLE`: (a) Use `UNAVAILABLE` if the client can
	// retry just the failing call. (b) Use `ABORTED` if the client should
	// retry at a higher level. For example, when a client-specified
	// test-and-set fails, indicating the client should restart a
	// read-modify-write sequence. (c) Use `FAILED_PRECONDITION` if the
	// client should not retry until the system state has been explicitly
	// fixed. For example, if an "rmdir" fails because the directory is
	// non-empty, `FAILED_PRECONDITION` should be returned since the client
	// should not retry unless the files are deleted from the directory.
	// HTTP Mapping: 400 Bad Request
	//   "ABORTED" - The operation was aborted, typically due to a
	// concurrency issue such as a sequencer check failure or transaction
	// abort. See the guidelines above for deciding between
	// `FAILED_PRECONDITION`, `ABORTED`, and `UNAVAILABLE`. HTTP Mapping:
	// 409 Conflict
	//   "OUT_OF_RANGE" - The operation was attempted past the valid range.
	// E.g., seeking or reading past end-of-file. Unlike `INVALID_ARGUMENT`,
	// this error indicates a problem that may be fixed if the system state
	// changes. For example, a 32-bit file system will generate
	// `INVALID_ARGUMENT` if asked to read at an offset that is not in the
	// range [0,2^32-1], but it will generate `OUT_OF_RANGE` if asked to
	// read from an offset past the current file size. There is a fair bit
	// of overlap between `FAILED_PRECONDITION` and `OUT_OF_RANGE`. We
	// recommend using `OUT_OF_RANGE` (the more specific error) when it
	// applies so that callers who are iterating through a space can easily
	// look for an `OUT_OF_RANGE` error to detect when they are done. HTTP
	// Mapping: 400 Bad Request
	//   "UNIMPLEMENTED" - The operation is not implemented or is not
	// supported/enabled in this service. HTTP Mapping: 501 Not Implemented
	//   "INTERNAL" - Internal errors. This means that some invariants
	// expected by the underlying system have been broken. This error code
	// is reserved for serious errors. HTTP Mapping: 500 Internal Server
	// Error
	//   "UNAVAILABLE" - The service is currently unavailable. This is most
	// likely a transient condition, which can be corrected by retrying with
	// a backoff. Note that it is not always safe to retry non-idempotent
	// operations. See the guidelines above for deciding between
	// `FAILED_PRECONDITION`, `ABORTED`, and `UNAVAILABLE`. HTTP Mapping:
	// 503 Service Unavailable
	//   "DATA_LOSS" - Unrecoverable data loss or corruption. HTTP Mapping:
	// 500 Internal Server Error
	StatusCode string `json:"statusCode,omitempty"`

	// UserFacingMessage: The message to send users about the status of
	// their request. If unset, a generic message based on the `status_code`
	// is sent.
	UserFacingMessage string `json:"userFacingMessage,omitempty"`

	// ForceSendFields is a list of field names (e.g. "StatusCode") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "StatusCode") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ActionStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ActionStatus
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Annotation: Annotations associated with the plain-text body of the
// message. Example plain-text message body: ``` Hello @FooBot how are
// you!" ``` The corresponding annotations metadata: ```
// "annotations":[{ "type":"USER_MENTION", "startIndex":6, "length":7,
// "userMention": { "user": { "name":"users/{user}",
// "displayName":"FooBot", "avatarUrl":"https://goo.gl/aeDtrS",
// "type":"BOT" }, "type":"MENTION" } }] ```
type Annotation struct {
	// Length: Length of the substring in the plain-text message body this
	// annotation corresponds to.
	Length int64 `json:"length,omitempty"`

	// SlashCommand: The metadata for a slash command.
	SlashCommand *SlashCommandMetadata `json:"slashCommand,omitempty"`

	// StartIndex: Start index (0-based, inclusive) in the plain-text
	// message body this annotation corresponds to.
	StartIndex int64 `json:"startIndex,omitempty"`

	// Type: The type of this annotation.
	//
	// Possible values:
	//   "ANNOTATION_TYPE_UNSPECIFIED" - Default value for the enum. DO NOT
	// USE.
	//   "USER_MENTION" - A user is mentioned.
	//   "SLASH_COMMAND" - A slash command is invoked.
	Type string `json:"type,omitempty"`

	// UserMention: The metadata of user mention.
	UserMention *UserMentionMetadata `json:"userMention,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Length") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Length") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Annotation) MarshalJSON() ([]byte, error) {
	type NoMethod Annotation
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Attachment: An attachment in Google Chat.
type Attachment struct {
	// AttachmentDataRef: A reference to the attachment data. This is used
	// with the media API to download the attachment data.
	AttachmentDataRef *AttachmentDataRef `json:"attachmentDataRef,omitempty"`

	// ContentName: The original file name for the content, not the full
	// path.
	ContentName string `json:"contentName,omitempty"`

	// ContentType: The content type (MIME type) of the file.
	ContentType string `json:"contentType,omitempty"`

	// DownloadUri: Output only. The download URL which should be used to
	// allow a human user to download the attachment. Chat apps should not
	// use this URL to download attachment content.
	DownloadUri string `json:"downloadUri,omitempty"`

	// DriveDataRef: A reference to the drive attachment. This is used with
	// the Drive API.
	DriveDataRef *DriveDataRef `json:"driveDataRef,omitempty"`

	// Name: Resource name of the attachment, in the form
	// "spaces/*/messages/*/attachments/*".
	Name string `json:"name,omitempty"`

	// Source: The source of the attachment.
	//
	// Possible values:
	//   "SOURCE_UNSPECIFIED"
	//   "DRIVE_FILE"
	//   "UPLOADED_CONTENT"
	Source string `json:"source,omitempty"`

	// ThumbnailUri: Output only. The thumbnail URL which should be used to
	// preview the attachment to a human user. Chat apps should not use this
	// URL to download attachment content.
	ThumbnailUri string `json:"thumbnailUri,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "AttachmentDataRef")
	// to unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AttachmentDataRef") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *Attachment) MarshalJSON() ([]byte, error) {
	type NoMethod Attachment
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type AttachmentDataRef struct {
	// ResourceName: The resource name of the attachment data. This is used
	// with the media API to download the attachment data.
	ResourceName string `json:"resourceName,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ResourceName") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ResourceName") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *AttachmentDataRef) MarshalJSON() ([]byte, error) {
	type NoMethod AttachmentDataRef
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Button: A button. Can be a text button or an image button.
type Button struct {
	// ImageButton: A button with image and onclick action.
	ImageButton *ImageButton `json:"imageButton,omitempty"`

	// TextButton: A button with text and onclick action.
	TextButton *TextButton `json:"textButton,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ImageButton") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ImageButton") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Button) MarshalJSON() ([]byte, error) {
	type NoMethod Button
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Card: A card is a UI element that can contain UI widgets such as
// texts, images.
type Card struct {
	// CardActions: The actions of this card.
	CardActions []*CardAction `json:"cardActions,omitempty"`

	// Header: The header of the card. A header usually contains a title and
	// an image.
	Header *CardHeader `json:"header,omitempty"`

	// Name: Name of the card.
	Name string `json:"name,omitempty"`

	// Sections: Sections are separated by a line divider.
	Sections []*Section `json:"sections,omitempty"`

	// ForceSendFields is a list of field names (e.g. "CardActions") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CardActions") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Card) MarshalJSON() ([]byte, error) {
	type NoMethod Card
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// CardAction: A card action is the action associated with the card. For
// an invoice card, a typical action would be: delete invoice, email
// invoice or open the invoice in browser. Not supported by Google Chat
// apps.
type CardAction struct {
	// ActionLabel: The label used to be displayed in the action menu item.
	ActionLabel string `json:"actionLabel,omitempty"`

	// OnClick: The onclick action for this action item.
	OnClick *OnClick `json:"onClick,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ActionLabel") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ActionLabel") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *CardAction) MarshalJSON() ([]byte, error) {
	type NoMethod CardAction
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type CardHeader struct {
	// ImageStyle: The image's type (e.g. square border or circular border).
	//
	// Possible values:
	//   "IMAGE_STYLE_UNSPECIFIED"
	//   "IMAGE" - Square border.
	//   "AVATAR" - Circular border.
	ImageStyle string `json:"imageStyle,omitempty"`

	// ImageUrl: The URL of the image in the card header.
	ImageUrl string `json:"imageUrl,omitempty"`

	// Subtitle: The subtitle of the card header.
	Subtitle string `json:"subtitle,omitempty"`

	// Title: The title must be specified. The header has a fixed height: if
	// both a title and subtitle is specified, each will take up 1 line. If
	// only the title is specified, it will take up both lines.
	Title string `json:"title,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ImageStyle") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ImageStyle") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *CardHeader) MarshalJSON() ([]byte, error) {
	type NoMethod CardHeader
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// CardWithId: Widgets for Chat apps to specify.
type CardWithId struct {
	// Card: Cards support a defined layout, interactive UI elements like
	// buttons, and rich media like images. Use this card to present
	// detailed information, gather information from users, and guide users
	// to take a next step.
	Card *GoogleAppsCardV1Card `json:"card,omitempty"`

	// CardId: Required for `cardsV2` messages. Chat app-specified
	// identifier for this widget. Scoped within a message.
	CardId string `json:"cardId,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Card") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Card") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *CardWithId) MarshalJSON() ([]byte, error) {
	type NoMethod CardWithId
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// ChatAppLogEntry: JSON payload of error messages. If the Cloud Logging
// API is enabled, these error messages are logged to Google Cloud
// Logging (https://cloud.google.com/logging/docs).
type ChatAppLogEntry struct {
	// Deployment: The deployment that caused the error. For Chat apps built
	// in Apps Script, this is the deployment ID defined by Apps Script.
	Deployment string `json:"deployment,omitempty"`

	// DeploymentFunction: The unencrypted `callback_method` name that was
	// running when the error was encountered.
	DeploymentFunction string `json:"deploymentFunction,omitempty"`

	// Error: The error code and message.
	Error *Status `json:"error,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Deployment") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Deployment") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ChatAppLogEntry) MarshalJSON() ([]byte, error) {
	type NoMethod ChatAppLogEntry
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Color: Represents a color in the RGBA color space. This
// representation is designed for simplicity of conversion to/from color
// representations in various languages over compactness. For example,
// the fields of this representation can be trivially provided to the
// constructor of `java.awt.Color` in Java; it can also be trivially
// provided to UIColor's `+colorWithRed:green:blue:alpha` method in iOS;
// and, with just a little work, it can be easily formatted into a CSS
// `rgba()` string in JavaScript. This reference page doesn't carry
// information about the absolute color space that should be used to
// interpret the RGB value (e.g. sRGB, Adobe RGB, DCI-P3, BT.2020,
// etc.). By default, applications should assume the sRGB color space.
// When color equality needs to be decided, implementations, unless
// documented otherwise, treat two colors as equal if all their red,
// green, blue, and alpha values each differ by at most 1e-5. Example
// (Java): import com.google.type.Color; // ... public static
// java.awt.Color fromProto(Color protocolor) { float alpha =
// protocolor.hasAlpha() ? protocolor.getAlpha().getValue() : 1.0;
// return new java.awt.Color( protocolor.getRed(),
// protocolor.getGreen(), protocolor.getBlue(), alpha); } public static
// Color toProto(java.awt.Color color) { float red = (float)
// color.getRed(); float green = (float) color.getGreen(); float blue =
// (float) color.getBlue(); float denominator = 255.0; Color.Builder
// resultBuilder = Color .newBuilder() .setRed(red / denominator)
// .setGreen(green / denominator) .setBlue(blue / denominator); int
// alpha = color.getAlpha(); if (alpha != 255) { result.setAlpha(
// FloatValue .newBuilder() .setValue(((float) alpha) / denominator)
// .build()); } return resultBuilder.build(); } // ... Example (iOS /
// Obj-C): // ... static UIColor* fromProto(Color* protocolor) { float
// red = [protocolor red]; float green = [protocolor green]; float blue
// = [protocolor blue]; FloatValue* alpha_wrapper = [protocolor alpha];
// float alpha = 1.0; if (alpha_wrapper != nil) { alpha = [alpha_wrapper
// value]; } return [UIColor colorWithRed:red green:green blue:blue
// alpha:alpha]; } static Color* toProto(UIColor* color) { CGFloat red,
// green, blue, alpha; if (![color getRed:&red green:&green blue:&blue
// alpha:&alpha]) { return nil; } Color* result = [[Color alloc] init];
// [result setRed:red]; [result setGreen:green]; [result setBlue:blue];
// if (alpha <= 0.9999) { [result
// setAlpha:floatWrapperWithValue(alpha)]; } [result autorelease];
// return result; } // ... Example (JavaScript): // ... var
// protoToCssColor = function(rgb_color) { var redFrac = rgb_color.red
// || 0.0; var greenFrac = rgb_color.green || 0.0; var blueFrac =
// rgb_color.blue || 0.0; var red = Math.floor(redFrac * 255); var green
// = Math.floor(greenFrac * 255); var blue = Math.floor(blueFrac * 255);
// if (!('alpha' in rgb_color)) { return rgbToCssColor(red, green,
// blue); } var alphaFrac = rgb_color.alpha.value || 0.0; var rgbParams
// = [red, green, blue].join(','); return ['rgba(', rgbParams, ',',
// alphaFrac, ')'].join(”); }; var rgbToCssColor = function(red, green,
// blue) { var rgbNumber = new Number((red << 16) | (green << 8) |
// blue); var hexString = rgbNumber.toString(16); var missingZeros = 6 -
// hexString.length; var resultBuilder = ['#']; for (var i = 0; i <
// missingZeros; i++) { resultBuilder.push('0'); }
// resultBuilder.push(hexString); return resultBuilder.join(”); }; //
// ...
type Color struct {
	// Alpha: The fraction of this color that should be applied to the
	// pixel. That is, the final pixel color is defined by the equation:
	// `pixel color = alpha * (this color) + (1.0 - alpha) * (background
	// color)` This means that a value of 1.0 corresponds to a solid color,
	// whereas a value of 0.0 corresponds to a completely transparent color.
	// This uses a wrapper message rather than a simple float scalar so that
	// it is possible to distinguish between a default value and the value
	// being unset. If omitted, this color object is rendered as a solid
	// color (as if the alpha value had been explicitly given a value of
	// 1.0).
	Alpha float64 `json:"alpha,omitempty"`

	// Blue: The amount of blue in the color as a value in the interval [0,
	// 1].
	Blue float64 `json:"blue,omitempty"`

	// Green: The amount of green in the color as a value in the interval
	// [0, 1].
	Green float64 `json:"green,omitempty"`

	// Red: The amount of red in the color as a value in the interval [0,
	// 1].
	Red float64 `json:"red,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Alpha") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Alpha") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Color) MarshalJSON() ([]byte, error) {
	type NoMethod Color
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

func (s *Color) UnmarshalJSON(data []byte) error {
	type NoMethod Color
	var s1 struct {
		Alpha gensupport.JSONFloat64 `json:"alpha"`
		Blue  gensupport.JSONFloat64 `json:"blue"`
		Green gensupport.JSONFloat64 `json:"green"`
		Red   gensupport.JSONFloat64 `json:"red"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Alpha = float64(s1.Alpha)
	s.Blue = float64(s1.Blue)
	s.Green = float64(s1.Green)
	s.Red = float64(s1.Red)
	return nil
}

// CommonEventObject: Represents information about the user's client,
// such as locale, host app, and platform. For Chat apps,
// `CommonEventObject` includes data submitted by users interacting with
// cards, like data entered in dialogs
// (https://developers.google.com/chat/how-tos/dialogs).
type CommonEventObject struct {
	// FormInputs: A map containing the current values of the widgets in a
	// card. The map keys are the string IDs assigned to each widget, and
	// the values represent inputs to the widget. Depending on the input
	// data type, a different object represents each input: For single-value
	// widgets, `StringInput`. For multi-value widgets, an array of
	// `StringInput` objects. For a date-time picker, a `DateTimeInput`. For
	// a date-only picker, a `DateInput`. For a time-only picker, a
	// `TimeInput`. Corresponds with the data entered by a user on a card in
	// a dialog (https://developers.google.com/chat/how-tos/dialogs).
	FormInputs map[string]Inputs `json:"formInputs,omitempty"`

	// HostApp: The hostApp enum which indicates the app the add-on is
	// invoked from. Always `CHAT` for Chat apps.
	//
	// Possible values:
	//   "UNSPECIFIED_HOST_APP" - Google can't identify a host app.
	//   "GMAIL" - The add-on launches from Gmail.
	//   "CALENDAR" - The add-on launches from Google Calendar.
	//   "DRIVE" - The add-on launches from Google Drive.
	//   "DEMO" - Not used.
	//   "DOCS" - The add-on launches from Google Docs.
	//   "MEET" - The add-on launches from Google Meet.
	//   "SHEETS" - The add-on launches from Google Sheets.
	//   "SLIDES" - The add-on launches from Google Slides.
	//   "DRAWINGS" - The add-on launches from Google Drawings.
	//   "CHAT" - A Google Chat app. Not used for Google Workspace Add-ons.
	HostApp string `json:"hostApp,omitempty"`

	// InvokedFunction: Name of the invoked function associated with the
	// widget. Only set for Chat apps.
	InvokedFunction string `json:"invokedFunction,omitempty"`

	// Parameters: Custom parameters
	// (/chat/api/reference/rest/v1/cards#ActionParameter) passed to the
	// invoked function. Both keys and values must be strings.
	Parameters map[string]string `json:"parameters,omitempty"`

	// Platform: The platform enum which indicates the platform where the
	// event originates (`WEB`, `IOS`, or `ANDROID`). Not supported by Chat
	// apps.
	//
	// Possible values:
	//   "UNKNOWN_PLATFORM"
	//   "WEB"
	//   "IOS"
	//   "ANDROID"
	Platform string `json:"platform,omitempty"`

	// TimeZone: The timezone ID and offset from Coordinated Universal Time
	// (UTC).
	TimeZone *TimeZone `json:"timeZone,omitempty"`

	// UserLocale: The full `locale.displayName` in the format of [ISO 639
	// language code]-[ISO 3166 country/region code] such as "en-US". Not
	// supported by Chat apps.
	UserLocale string `json:"userLocale,omitempty"`

	// ForceSendFields is a list of field names (e.g. "FormInputs") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "FormInputs") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *CommonEventObject) MarshalJSON() ([]byte, error) {
	type NoMethod CommonEventObject
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DateInput: Date input values. Not supported by Chat apps.
type DateInput struct {
	// MsSinceEpoch: Time since epoch time, in milliseconds.
	MsSinceEpoch int64 `json:"msSinceEpoch,omitempty,string"`

	// ForceSendFields is a list of field names (e.g. "MsSinceEpoch") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "MsSinceEpoch") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DateInput) MarshalJSON() ([]byte, error) {
	type NoMethod DateInput
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DateTimeInput: Date and time input values. Not supported by Chat
// apps.
type DateTimeInput struct {
	// HasDate: Whether the `datetime` input includes a calendar date.
	HasDate bool `json:"hasDate,omitempty"`

	// HasTime: Whether the `datetime` input includes a timestamp.
	HasTime bool `json:"hasTime,omitempty"`

	// MsSinceEpoch: Time since epoch time, in milliseconds.
	MsSinceEpoch int64 `json:"msSinceEpoch,omitempty,string"`

	// ForceSendFields is a list of field names (e.g. "HasDate") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "HasDate") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DateTimeInput) MarshalJSON() ([]byte, error) {
	type NoMethod DateTimeInput
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DeprecatedEvent: Google Chat events.
type DeprecatedEvent struct {
	// Action: The form action data associated with an interactive card that
	// was clicked. Only populated for CARD_CLICKED events. See the
	// Interactive Cards guide (/chat/how-tos/cards-onclick) for more
	// information.
	Action *FormAction `json:"action,omitempty"`

	// Common: Represents information about the user's client, such as
	// locale, host app, and platform. For Chat apps, `CommonEventObject`
	// includes information submitted by users interacting with dialogs
	// (https://developers.google.com/chat/how-tos/dialogs), like data
	// entered on a card.
	Common *CommonEventObject `json:"common,omitempty"`

	// ConfigCompleteRedirectUrl: The URL the Chat app should redirect the
	// user to after they have completed an authorization or configuration
	// flow outside of Google Chat. For more information, see Connect a Chat
	// app with other services & tools
	// (https://developers.google.com/chat/how-tos/connect-web-services-tools).
	ConfigCompleteRedirectUrl string `json:"configCompleteRedirectUrl,omitempty"`

	// DialogEventType: The type of dialog
	// (https://developers.google.com/chat/how-tos/dialogs) event received.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - This could be used when the corresponding
	// event is not dialog related. For example an @mention.
	//   "REQUEST_DIALOG" - Any user action that opens a
	// [dialog](https://developers.google.com/chat/how-tos/dialogs).
	//   "SUBMIT_DIALOG" - A card click event from a
	// [dialog](https://developers.google.com/chat/how-tos/dialogs).
	//   "CANCEL_DIALOG" - The
	// [dialog](https://developers.google.com/chat/how-tos/dialogs) was
	// cancelled.
	DialogEventType string `json:"dialogEventType,omitempty"`

	// EventTime: The timestamp indicating when the event occurred.
	EventTime string `json:"eventTime,omitempty"`

	// IsDialogEvent: True when the event is related to dialogs
	// (https://developers.google.com/chat/how-tos/dialogs).
	IsDialogEvent bool `json:"isDialogEvent,omitempty"`

	// Message: The message that triggered the event, if applicable.
	Message *Message `json:"message,omitempty"`

	// Space: The space in which the event occurred.
	Space *Space `json:"space,omitempty"`

	// ThreadKey: The Chat app-defined key for the thread related to the
	// event. See `spaces.messages.thread.threadKey`
	// (/chat/api/reference/rest/v1/spaces.messages#Thread.FIELDS.thread_key)
	//  for more information.
	ThreadKey string `json:"threadKey,omitempty"`

	// Token: A secret value that legacy Chat apps can use to verify if a
	// request is from Google. Google randomly generates the token, and its
	// value remains static. You can obtain, revoke, or regenerate the token
	// from the Chat API configuration page
	// (https://console.cloud.google.com/apis/api/chat.googleapis.com/hangouts-chat)
	// in the Google Cloud Console. Modern Chat apps don't use this field.
	// It is absent from API responses and the Chat API configuration page
	// (https://console.cloud.google.com/apis/api/chat.googleapis.com/hangouts-chat).
	Token string `json:"token,omitempty"`

	// Type: The type of the event.
	//
	// Possible values:
	//   "UNSPECIFIED" - Default value for the enum. DO NOT USE.
	//   "MESSAGE" - A message was sent in a space.
	//   "ADDED_TO_SPACE" - The Chat app was added to a space by a Chat user
	// or Workspace administrator.
	//   "REMOVED_FROM_SPACE" - The Chat app was removed from a space by a
	// Chat user or Workspace administrator.
	//   "CARD_CLICKED" - The Chat app's interactive card was clicked.
	Type string `json:"type,omitempty"`

	// User: The user that triggered the event.
	User *User `json:"user,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Action") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DeprecatedEvent) MarshalJSON() ([]byte, error) {
	type NoMethod DeprecatedEvent
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Dialog: Wrapper around the card body of the dialog.
type Dialog struct {
	// Body: Input only. Body of the dialog, which is rendered in a modal.
	// Google Chat apps do not support the following card entities:
	// `DateTimePicker`, `OnChangeAction`.
	Body *GoogleAppsCardV1Card `json:"body,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Body") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Body") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Dialog) MarshalJSON() ([]byte, error) {
	type NoMethod Dialog
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DialogAction: Contains a dialog
// (https://developers.google.com/chat/how-tos/dialogs) and request
// status code.
type DialogAction struct {
	// ActionStatus: Input only. Status for a request to either invoke or
	// submit a dialog (https://developers.google.com/chat/how-tos/dialogs).
	// Displays a status and message to users, if necessary. For example, in
	// case of an error or success.
	ActionStatus *ActionStatus `json:"actionStatus,omitempty"`

	// Dialog: Input only. Dialog
	// (https://developers.google.com/chat/how-tos/dialogs) for the request.
	Dialog *Dialog `json:"dialog,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ActionStatus") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ActionStatus") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DialogAction) MarshalJSON() ([]byte, error) {
	type NoMethod DialogAction
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DriveDataRef: A reference to the data of a drive attachment.
type DriveDataRef struct {
	// DriveFileId: The id for the drive file, for use with the Drive API.
	DriveFileId string `json:"driveFileId,omitempty"`

	// ForceSendFields is a list of field names (e.g. "DriveFileId") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DriveFileId") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DriveDataRef) MarshalJSON() ([]byte, error) {
	type NoMethod DriveDataRef
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use
// it as the request or the response type of an API method. For
// instance: service Foo { rpc Bar(google.protobuf.Empty) returns
// (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`
}

// FormAction: A form action describes the behavior when the form is
// submitted. For example, an Apps Script can be invoked to handle the
// form.
type FormAction struct {
	// ActionMethodName: The method name is used to identify which part of
	// the form triggered the form submission. This information is echoed
	// back to the Chat app as part of the card click event. The same method
	// name can be used for several elements that trigger a common behavior
	// if desired.
	ActionMethodName string `json:"actionMethodName,omitempty"`

	// Parameters: List of action parameters.
	Parameters []*ActionParameter `json:"parameters,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ActionMethodName") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ActionMethodName") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *FormAction) MarshalJSON() ([]byte, error) {
	type NoMethod FormAction
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Action: An action that describes the behavior when
// the form is submitted. For example, an Apps Script can be invoked to
// handle the form. If the action is triggered, the form values are sent
// to the server.
type GoogleAppsCardV1Action struct {
	// Function: A custom function to invoke when the containing element is
	// clicked or othrwise activated. For example usage, see Create
	// interactive cards
	// (https://developers.google.com/chat/how-tos/cards-onclick).
	Function string `json:"function,omitempty"`

	// Interaction: Optional. Required when opening a dialog
	// (https://developers.google.com/chat/how-tos/dialogs). What to do in
	// response to an interaction with a user, such as a user clicking
	// button on a card message. If unspecified, the app responds by
	// executing an `action` - like opening a link or running a function -
	// as normal. By specifying an `interaction`, the app can respond in
	// special interactive ways. For example, by setting `interaction` to
	// `OPEN_DIALOG`, the app can open a dialog
	// (https://developers.google.com/chat/how-tos/dialogs). When specified,
	// a loading indicator is not shown. Supported by Chat apps, but not
	// Google Workspace Add-ons. If specified for an add-on, the entire card
	// is stripped and nothing is shown in the client.
	//
	// Possible values:
	//   "INTERACTION_UNSPECIFIED" - Default value. The `action` executes as
	// normal.
	//   "OPEN_DIALOG" - Opens a
	// [dialog](https://developers.google.com/chat/how-tos/dialogs), a
	// windowed, card-based interface that Chat apps use to interact with
	// users. Only supported by Chat apps in response to button-clicks on
	// card messages. Not supported by Google Workspace Add-ons. If
	// specified for an add-on, the entire card is stripped and nothing is
	// shown in the client.
	Interaction string `json:"interaction,omitempty"`

	// LoadIndicator: Specifies the loading indicator that the action
	// displays while making the call to the action.
	//
	// Possible values:
	//   "SPINNER" - Displays a spinner to indicate that content is loading.
	//   "NONE" - Nothing is displayed.
	LoadIndicator string `json:"loadIndicator,omitempty"`

	// Parameters: List of action parameters.
	Parameters []*GoogleAppsCardV1ActionParameter `json:"parameters,omitempty"`

	// PersistValues: Indicates whether form values persist after the
	// action. The default value is `false`. If `true`, form values remain
	// after the action is triggered. To let the user make changes while the
	// action is being processed, set LoadIndicator
	// (https://developers.google.com/workspace/add-ons/reference/rpc/google.apps.card.v1#loadindicator)
	// to `NONE`. For card messages
	// (https://developers.google.com/chat/api/guides/message-formats/cards)
	// in Chat apps, you must also set the action's ResponseType
	// (https://developers.google.com/chat/api/reference/rest/v1/spaces.messages#responsetype)
	// to `UPDATE_MESSAGE` and use the same `card_id`
	// (https://developers.google.com/chat/api/reference/rest/v1/spaces.messages#CardWithId)
	// from the card that contained the action. If `false`, the form values
	// are cleared when the action is triggered. To prevent the user from
	// making changes while the action is being processed, set LoadIndicator
	// (https://developers.google.com/workspace/add-ons/reference/rpc/google.apps.card.v1#loadindicator)
	// to `SPINNER`.
	PersistValues bool `json:"persistValues,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Function") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Function") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Action) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Action
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ActionParameter: List of string parameters to supply
// when the action method is invoked. For example, consider three snooze
// buttons: snooze now, snooze 1 day, snooze next week. You might use
// action method = snooze(), passing the snooze type and snooze time in
// the list of string parameters. To learn more, see CommonEventObject
// (https://developers.google.com/chat/api/reference/rest/v1/Event#commoneventobject).
type GoogleAppsCardV1ActionParameter struct {
	// Key: The name of the parameter for the action script.
	Key string `json:"key,omitempty"`

	// Value: The value of the parameter.
	Value string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Key") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Key") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1ActionParameter) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ActionParameter
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1BorderStyle: Represents the complete border style
// applied to items in a widget.
type GoogleAppsCardV1BorderStyle struct {
	// CornerRadius: The corner radius for the border.
	CornerRadius int64 `json:"cornerRadius,omitempty"`

	// StrokeColor: The colors to use when the type is `BORDER_TYPE_STROKE`.
	StrokeColor *Color `json:"strokeColor,omitempty"`

	// Type: The border type.
	//
	// Possible values:
	//   "BORDER_TYPE_UNSPECIFIED" - No value specified.
	//   "NO_BORDER" - Default value. No border.
	//   "STROKE" - Outline.
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "CornerRadius") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CornerRadius") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1BorderStyle) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1BorderStyle
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Button: A text, icon, or text + icon button that
// users can click. To make an image a clickable button, specify an
// Image (not an ImageComponent) and set an `onClick` action. Currently
// supported in Chat apps (including [dialogs]
// (https://developers.google.com/chat/how-tos/dialogs) and [card
// messages]
// (https://developers.google.com/chat/api/guides/message-formats/cards))
// and Google Workspace Add-ons.
type GoogleAppsCardV1Button struct {
	// AltText: The alternative text used for accessibility. Set descriptive
	// text that lets users know what the button does. For example, if a
	// button opens a hyperlink, you might write: "Opens a new browser tab
	// and navigates to the Google Chat developer documentation at
	// https://developers.google.com/chat".
	AltText string `json:"altText,omitempty"`

	// Color: If set, the button is filled with a solid background color and
	// the font color changes to maintain contrast with the background
	// color. For example, setting a blue background will likely result in
	// white text. If unset, the image background is white and the font
	// color is blue. For red, green and blue, the value of each field is a
	// `float` number that can be expressed in either of two ways: as a
	// number between 0 and 255 divided by 255 (153/255) or as a value
	// between 0 and 1 (0.6). 0 represents the absence of a color and 1 or
	// 255/255 represent the full presence of that color on the RGB scale.
	// Optionally set alpha, which sets a level of transparency using this
	// equation: ``` pixel color = alpha * (this color) + (1.0 - alpha) *
	// (background color) ``` For alpha, a value of 1 corresponds with a
	// solid color, and a value of 0 corresponds with a completely
	// transparent color. For example, the following color represents a half
	// transparent red: ``` "color": { "red": 1, "green": 0, "blue": 0,
	// "alpha": 0.5 } ```
	Color *Color `json:"color,omitempty"`

	// Disabled: If `true`, the button is displayed in an inactive state and
	// doesn't respond to user actions.
	Disabled bool `json:"disabled,omitempty"`

	// Icon: The icon image. If both `icon` and `text` are set, then the
	// icon appears before the text.
	Icon *GoogleAppsCardV1Icon `json:"icon,omitempty"`

	// OnClick: Required. The action to perform when the button is clicked,
	// such as opening a hyperlink or running a custom function.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`

	// Text: The text displayed inside the button.
	Text string `json:"text,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AltText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AltText") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Button) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Button
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ButtonList: A list of buttons layed out horizontally.
type GoogleAppsCardV1ButtonList struct {
	// Buttons: An array of buttons.
	Buttons []*GoogleAppsCardV1Button `json:"buttons,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Buttons") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Buttons") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1ButtonList) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ButtonList
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Card: Cards support a defined layout, interactive UI
// elements like buttons, and rich media like images. Use cards to
// present detailed information, gather information from users, and
// guide users to take a next step. In Google Chat, cards appear in
// several places: - As stand-alone messages. - Accompanying a text
// message, just beneath the text message. - As a dialog
// (https://developers.google.com/chat/how-tos/dialogs). The following
// example JSON creates a "contact card" that features: - A header with
// the contact's name, job title, avatar picture. - A section with the
// contact information, including formatted text. - Buttons that users
// can click to share the contact or see more or less info. !Example
// contact card
// (https://developers.google.com/chat/images/card_api_reference.png)
// ``` { "cardsV2": [ { "cardId": "unique-card-id", "card": { "header":
// { "title": "Sasha", "subtitle": "Software Engineer", "imageUrl":
// "https://developers.google.com/chat/images/quickstart-app-avatar.png",
//
//	"imageType": "CIRCLE", "imageAltText": "Avatar for Sasha", },
//
// "sections": [ { "header": "Contact Info", "collapsible": true,
// "uncollapsibleWidgetsCount": 1, "widgets": [ { "decoratedText": {
// "startIcon": { "knownIcon": "EMAIL", }, "text": "sasha@example.com",
// } }, { "decoratedText": { "startIcon": { "knownIcon": "PERSON", },
// "text": "Online", }, }, { "decoratedText": { "startIcon": {
// "knownIcon": "PHONE", }, "text": "+1 (555) 555-1234", } }, {
// "buttonList": { "buttons": [ { "text": "Share", "onClick": {
// "openLink": { "url": "https://example.com/share", } } }, { "text":
// "Edit", "onClick": { "action": { "function": "goToView",
// "parameters": [ { "key": "viewType", "value": "EDIT", } ], } } }, ],
// } }, ], }, ], }, } ], } ```
type GoogleAppsCardV1Card struct {
	// CardActions: The card's actions. Actions are added to the card's
	// toolbar menu. Because Chat app cards have no toolbar, `cardActions[]`
	// is not supported by Chat apps. For example, the following JSON
	// constructs a card action menu with Settings and Send Feedback
	// options: ``` "card_actions": [ { "actionLabel": "Settings",
	// "onClick": { "action": { "functionName": "goToView", "parameters": [
	// { "key": "viewType", "value": "SETTING" } ], "loadIndicator":
	// "LoadIndicator.SPINNER" } } }, { "actionLabel": "Send Feedback",
	// "onClick": { "openLink": { "url": "https://example.com/feedback" } }
	// } ] ```
	CardActions []*GoogleAppsCardV1CardAction `json:"cardActions,omitempty"`

	// DisplayStyle: In Google Workspace add-ons, sets the display
	// properties of the `peekCardHeader`. Not supported by Chat apps.
	//
	// Possible values:
	//   "DISPLAY_STYLE_UNSPECIFIED" - Do not use.
	//   "PEEK" - The header of the card appears at the bottom of the
	// sidebar, partially covering the current top card of the stack.
	// Clicking the header pops the card into the card stack. If the card
	// has no header, a generated header is used instead.
	//   "REPLACE" - Default value. The card is shown by replacing the view
	// of the top card in the card stack.
	DisplayStyle string `json:"displayStyle,omitempty"`

	// FixedFooter: The fixed footer shown at the bottom of this card.
	// Setting `fixedFooter` without specifying a `primaryButton` or a
	// `secondaryButton` causes an error. Chat apps support `fixedFooter` in
	// dialogs (https://developers.google.com/chat/how-tos/dialogs), but not
	// in card messages
	// (https://developers.google.com/chat/api/guides/message-formats/cards).
	FixedFooter *GoogleAppsCardV1CardFixedFooter `json:"fixedFooter,omitempty"`

	// Header: The header of the card. A header usually contains a leading
	// image and a title. Headers always appear at the top of a card.
	Header *GoogleAppsCardV1CardHeader `json:"header,omitempty"`

	// Name: Name of the card. Used as a card identifier in card navigation.
	// Because Chat apps don't support card navigation, they ignore this
	// field.
	Name string `json:"name,omitempty"`

	// PeekCardHeader: When displaying contextual content, the peek card
	// header acts as a placeholder so that the user can navigate forward
	// between the homepage cards and the contextual cards. Not supported by
	// Chat apps.
	PeekCardHeader *GoogleAppsCardV1CardHeader `json:"peekCardHeader,omitempty"`

	// Sections: Contains a collection of widgets. Each section has its own,
	// optional header. Sections are visually separated by a line divider.
	Sections []*GoogleAppsCardV1Section `json:"sections,omitempty"`

	// ForceSendFields is a list of field names (e.g. "CardActions") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CardActions") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Card) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Card
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CardAction: A card action is the action associated
// with the card. For example, an invoice card might include actions
// such as delete invoice, email invoice, or open the invoice in a
// browser. Not supported by Chat apps.
type GoogleAppsCardV1CardAction struct {
	// ActionLabel: The label that displays as the action menu item.
	ActionLabel string `json:"actionLabel,omitempty"`

	// OnClick: The `onClick` action for this action item.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ActionLabel") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ActionLabel") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1CardAction) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CardAction
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CardFixedFooter: A persistent (sticky) footer that
// that appears at the bottom of the card. Setting `fixedFooter` without
// specifying a `primaryButton` or a `secondaryButton` causes an error.
// Chat apps support `fixedFooter` in dialogs
// (https://developers.google.com/chat/how-tos/dialogs), but not in card
// messages
// (https://developers.google.com/chat/api/guides/message-formats/cards).
type GoogleAppsCardV1CardFixedFooter struct {
	// PrimaryButton: The primary button of the fixed footer. The button
	// must be a text button with text and color set.
	PrimaryButton *GoogleAppsCardV1Button `json:"primaryButton,omitempty"`

	// SecondaryButton: The secondary button of the fixed footer. The button
	// must be a text button with text and color set. `primaryButton` must
	// be set if `secondaryButton` is set.
	SecondaryButton *GoogleAppsCardV1Button `json:"secondaryButton,omitempty"`

	// ForceSendFields is a list of field names (e.g. "PrimaryButton") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "PrimaryButton") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1CardFixedFooter) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CardFixedFooter
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1CardHeader: Represents a card header.
type GoogleAppsCardV1CardHeader struct {
	// ImageAltText: The alternative text of this image which is used for
	// accessibility.
	ImageAltText string `json:"imageAltText,omitempty"`

	// ImageType: The shape used to crop the image.
	//
	// Possible values:
	//   "SQUARE" - Default value. Applies a square mask to the image. For
	// example, a 4x3 image becomes 3x3.
	//   "CIRCLE" - Applies a circular mask to the image. For example, a 4x3
	// image becomes a circle with a diameter of 3.
	ImageType string `json:"imageType,omitempty"`

	// ImageUrl: The HTTPS URL of the image in the card header.
	ImageUrl string `json:"imageUrl,omitempty"`

	// Subtitle: The subtitle of the card header. If specified, appears on
	// its own line below the `title`.
	Subtitle string `json:"subtitle,omitempty"`

	// Title: Required. The title of the card header. The header has a fixed
	// height: if both a title and subtitle are specified, each takes up one
	// line. If only the title is specified, it takes up both lines.
	Title string `json:"title,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ImageAltText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ImageAltText") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1CardHeader) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1CardHeader
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1DateTimePicker: Lets users specify a date, a time, or
// both a date and a time. Accepts text input from users, but features
// an interactive date and time selector that helps users enter
// correctly-formatted dates and times. If users enter a date or time
// incorrectly, the widget shows an error that prompts users to enter
// the correct format. Not supported by Chat apps. Support by Chat apps
// coming soon.
type GoogleAppsCardV1DateTimePicker struct {
	// Label: The text that prompts users to enter a date, time, or
	// datetime. Specify text that helps the user enter the information your
	// app needs. For example, if users are setting an appointment, then a
	// label like "Appointment date" or "Appointment date and time" might
	// work well.
	Label string `json:"label,omitempty"`

	// Name: The name by which the datetime picker is identified in a form
	// input event. For details about working with form inputs, see Receive
	// form data
	// (https://developers.google.com/chat/how-tos/dialogs#receive_form_data_from_dialogs).
	Name string `json:"name,omitempty"`

	// OnChangeAction: Triggered when the user clicks **Save** or **Clear**
	// from the datetime picker interface.
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`

	// TimezoneOffsetDate: The number representing the time zone offset from
	// UTC, in minutes. If set, the `value_ms_epoch` is displayed in the
	// specified time zone. If not set, it uses the user's time zone setting
	// on the client side.
	TimezoneOffsetDate int64 `json:"timezoneOffsetDate,omitempty"`

	// Type: What kind of date and time input the datetime picker supports.
	//
	// Possible values:
	//   "DATE_AND_TIME" - The user can select a date and time.
	//   "DATE_ONLY" - The user can only select a date.
	//   "TIME_ONLY" - The user can only select a time.
	Type string `json:"type,omitempty"`

	// ValueMsEpoch: The value displayed as the default value before user
	// input or previous user input, represented in milliseconds (Epoch time
	// (https://en.wikipedia.org/wiki/Unix_time)). For `DATE_AND_TIME` type,
	// the full epoch value is used. For `DATE_ONLY` type, only date of the
	// epoch time is used. For `TIME_ONLY` type, only time of the epoch time
	// is used. For example, to represent 3:00 AM, set epoch time to `3 * 60
	// * 60 * 1000`.
	ValueMsEpoch int64 `json:"valueMsEpoch,omitempty,string"`

	// ForceSendFields is a list of field names (e.g. "Label") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Label") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1DateTimePicker) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1DateTimePicker
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1DecoratedText: A widget that displays text with
// optional decorations such as a label above or below the text, an icon
// in front of the text, a selection widget or a button after the text.
type GoogleAppsCardV1DecoratedText struct {
	// BottomLabel: The text that appears below `text`. Always truncates.
	BottomLabel string `json:"bottomLabel,omitempty"`

	// Button: A button that can be clicked to trigger an action.
	Button *GoogleAppsCardV1Button `json:"button,omitempty"`

	// EndIcon: An icon displayed after the text. Supports built-in
	// (https://developers.google.com/chat/api/guides/message-formats/cards#builtinicons)
	// and custom
	// (https://developers.google.com/chat/api/guides/message-formats/cards#customicons)
	// icons.
	EndIcon *GoogleAppsCardV1Icon `json:"endIcon,omitempty"`

	// Icon: Deprecated in favor of `startIcon`.
	Icon *GoogleAppsCardV1Icon `json:"icon,omitempty"`

	// OnClick: When users click on `topLabel` or `bottomLabel`, this action
	// triggers.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`

	// StartIcon: The icon displayed in front of the text.
	StartIcon *GoogleAppsCardV1Icon `json:"startIcon,omitempty"`

	// SwitchControl: A switch widget can be clicked to change its state and
	// trigger an action.
	SwitchControl *GoogleAppsCardV1SwitchControl `json:"switchControl,omitempty"`

	// Text: Required. The primary text. Supports simple formatting. See
	// Text formatting for formatting details.
	Text string `json:"text,omitempty"`

	// TopLabel: The text that appears above `text`. Always truncates.
	TopLabel string `json:"topLabel,omitempty"`

	// WrapText: The wrap text setting. If `true`, the text wraps and
	// displays on multiple lines. Otherwise, the text is truncated. Only
	// applies to `text`, not `topLabel` and `bottomLabel`.
	WrapText bool `json:"wrapText,omitempty"`

	// ForceSendFields is a list of field names (e.g. "BottomLabel") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "BottomLabel") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1DecoratedText) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1DecoratedText
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Divider: Displays a divider between widgets, a
// horizontal line. For example, the following JSON creates a divider:
// ``` "divider": {} ```
type GoogleAppsCardV1Divider struct {
}

// GoogleAppsCardV1Grid: Displays a grid with a collection of items. A
// grid supports any number of columns and items. The number of rows is
// determined by items divided by columns. A grid with 10 items and 2
// columns has 5 rows. A grid with 11 items and 2 columns has 6 rows.
// For example, the following JSON creates a 2 column grid with a single
// item: ``` "grid": { "title": "A fine collection of items",
// "columnCount": 2, "borderStyle": { "type": "STROKE", "cornerRadius":
// 4 }, "items": [ { "image": { "imageUri":
// "https://www.example.com/image.png", "cropStyle": { "type": "SQUARE"
// }, "borderStyle": { "type": "STROKE" } }, "title": "An item",
// "textAlignment": "CENTER" } ], "onClick": { "openLink": { "url":
// "https://www.example.com" } } } ```
type GoogleAppsCardV1Grid struct {
	// BorderStyle: The border style to apply to each grid item.
	BorderStyle *GoogleAppsCardV1BorderStyle `json:"borderStyle,omitempty"`

	// ColumnCount: The number of columns to display in the grid. A default
	// value is used if this field isn't specified, and that default value
	// is different depending on where the grid is shown (dialog versus
	// companion).
	ColumnCount int64 `json:"columnCount,omitempty"`

	// Items: The items to display in the grid.
	Items []*GoogleAppsCardV1GridItem `json:"items,omitempty"`

	// OnClick: This callback is reused by each individual grid item, but
	// with the item's identifier and index in the items list added to the
	// callback's parameters.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`

	// Title: The text that displays in the grid header.
	Title string `json:"title,omitempty"`

	// ForceSendFields is a list of field names (e.g. "BorderStyle") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "BorderStyle") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Grid) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Grid
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1GridItem: Represents a single item in the grid
// layout.
type GoogleAppsCardV1GridItem struct {
	// Id: A user-specified identifier for this grid item. This identifier
	// is returned in the parent Grid's onClick callback parameters.
	Id string `json:"id,omitempty"`

	// Image: The image that displays in the grid item.
	Image *GoogleAppsCardV1ImageComponent `json:"image,omitempty"`

	// Layout: The layout to use for the grid item.
	//
	// Possible values:
	//   "GRID_ITEM_LAYOUT_UNSPECIFIED" - No layout specified.
	//   "TEXT_BELOW" - The title and subtitle are shown below the grid
	// item's image.
	//   "TEXT_ABOVE" - The title and subtitle are shown above the grid
	// item's image.
	Layout string `json:"layout,omitempty"`

	// Subtitle: The grid item's subtitle.
	Subtitle string `json:"subtitle,omitempty"`

	// Title: The grid item's title.
	Title string `json:"title,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Id") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Id") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1GridItem) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1GridItem
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Icon: An icon displayed in a widget on a card.
// Supports built-in
// (https://developers.google.com/chat/api/guides/message-formats/cards#builtinicons)
// and custom
// (https://developers.google.com/chat/api/guides/message-formats/cards#customicons)
// icons.
type GoogleAppsCardV1Icon struct {
	// AltText: Optional. A description of the icon used for accessibility.
	// If unspecified, the default value "Button" is provided. As a best
	// practice, you should set a helpful description for what the icon
	// displays, and if applicable, what it does. For example, `A user's
	// account portrait`, or `Opens a new browser tab and navigates to the
	// Google Chat developer documentation at
	// https://developers.google.com/chat`. If the icon is set in a Button,
	// the `altText` appears as helper text when the user hovers over the
	// button. However, if the button also sets `text`, the icon's `altText`
	// is ignored.
	AltText string `json:"altText,omitempty"`

	// IconUrl: Display a custom icon hosted at an HTTPS URL. For example:
	// ``` "iconUrl":
	// "https://developers.google.com/chat/images/quickstart-app-avatar.png"
	// ``` Supported file types include `.png` and `.jpg`.
	IconUrl string `json:"iconUrl,omitempty"`

	// ImageType: The crop style applied to the image. In some cases,
	// applying a `CIRCLE` crop causes the image to be drawn larger than a
	// built-in icon.
	//
	// Possible values:
	//   "SQUARE" - Default value. Applies a square mask to the image. For
	// example, a 4x3 image becomes 3x3.
	//   "CIRCLE" - Applies a circular mask to the image. For example, a 4x3
	// image becomes a circle with a diameter of 3.
	ImageType string `json:"imageType,omitempty"`

	// KnownIcon: Display one of the built-in icons provided by Google
	// Workspace. For example, to display an airplane icon, specify
	// `AIRPLANE`. For a bus, specify `BUS`. For a full list of supported
	// icons, see built-in icons
	// (https://developers.google.com/chat/api/guides/message-formats/cards#builtinicons).
	KnownIcon string `json:"knownIcon,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AltText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AltText") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Icon) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Icon
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Image: An image that is specified by a URL and can
// have an `onClick` action.
type GoogleAppsCardV1Image struct {
	// AltText: The alternative text of this image, used for accessibility.
	AltText string `json:"altText,omitempty"`

	// ImageUrl: The `https` URL that hosts the image. For example: ```
	// https://developers.google.com/chat/images/quickstart-app-avatar.png
	// ```
	ImageUrl string `json:"imageUrl,omitempty"`

	// OnClick: When a user clicks on the image, the click triggers this
	// action.
	OnClick *GoogleAppsCardV1OnClick `json:"onClick,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AltText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AltText") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Image) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Image
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ImageComponent: Represents an image.
type GoogleAppsCardV1ImageComponent struct {
	// AltText: The accessibility label for the image.
	AltText string `json:"altText,omitempty"`

	// BorderStyle: The border style to apply to the image.
	BorderStyle *GoogleAppsCardV1BorderStyle `json:"borderStyle,omitempty"`

	// CropStyle: The crop style to apply to the image.
	CropStyle *GoogleAppsCardV1ImageCropStyle `json:"cropStyle,omitempty"`

	// ImageUri: The image URL.
	ImageUri string `json:"imageUri,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AltText") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AltText") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1ImageComponent) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ImageComponent
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1ImageCropStyle: Represents the crop style applied to
// an image. For example, here's how to apply a 16 by 9 aspect ratio:
// ``` cropStyle { "type": "RECTANGLE_CUSTOM", "aspectRatio": 16/9 } ```
type GoogleAppsCardV1ImageCropStyle struct {
	// AspectRatio: The aspect ratio to use if the crop type is
	// `RECTANGLE_CUSTOM`. For example, here's how to apply a 16 by 9 aspect
	// ratio: ``` cropStyle { "type": "RECTANGLE_CUSTOM", "aspectRatio":
	// 16/9 } ```
	AspectRatio float64 `json:"aspectRatio,omitempty"`

	// Type: The crop type.
	//
	// Possible values:
	//   "IMAGE_CROP_TYPE_UNSPECIFIED" - No value specified. Do not use.
	//   "SQUARE" - Default value. Applies a square crop.
	//   "CIRCLE" - Applies a circular crop.
	//   "RECTANGLE_CUSTOM" - Applies a rectangular crop with a custom
	// aspect ratio. Set the custom aspect ratio with `aspectRatio`.
	//   "RECTANGLE_4_3" - Applies a rectangular crop with a 4:3 aspect
	// ratio.
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AspectRatio") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AspectRatio") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1ImageCropStyle) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1ImageCropStyle
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

func (s *GoogleAppsCardV1ImageCropStyle) UnmarshalJSON(data []byte) error {
	type NoMethod GoogleAppsCardV1ImageCropStyle
	var s1 struct {
		AspectRatio gensupport.JSONFloat64 `json:"aspectRatio"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.AspectRatio = float64(s1.AspectRatio)
	return nil
}

// GoogleAppsCardV1OnClick: Represents how to respond when users click
// an interactive element on a card, such as a button.
type GoogleAppsCardV1OnClick struct {
	// Action: If specified, an action is triggered by this `onClick`.
	Action *GoogleAppsCardV1Action `json:"action,omitempty"`

	// Card: A new card is pushed to the card stack after clicking if
	// specified. Supported by Google Workspace Add-ons, but not Chat apps.
	Card *GoogleAppsCardV1Card `json:"card,omitempty"`

	// OpenDynamicLinkAction: An add-on triggers this action when the action
	// needs to open a link. This differs from the `open_link` above in that
	// this needs to talk to server to get the link. Thus some preparation
	// work is required for web client to do before the open link action
	// response comes back.
	OpenDynamicLinkAction *GoogleAppsCardV1Action `json:"openDynamicLinkAction,omitempty"`

	// OpenLink: If specified, this `onClick` triggers an open link action.
	OpenLink *GoogleAppsCardV1OpenLink `json:"openLink,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Action") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1OnClick) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1OnClick
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1OpenLink: Represents an `onClick` event that opens a
// hyperlink.
type GoogleAppsCardV1OpenLink struct {
	// OnClose: Whether the client forgets about a link after opening it, or
	// observes it until the window closes. Not supported by Chat apps.
	//
	// Possible values:
	//   "NOTHING" - Default value. The card does not reload; nothing
	// happens.
	//   "RELOAD" - Reloads the card after the child window closes. If used
	// in conjunction with
	// [OpenAs.OVERLAY](https://developers.google.com/workspace/add-ons/refer
	// ence/rpc/google.apps.card.v1#openas), the child window acts as a
	// modal dialog and the parent card is blocked until the child window
	// closes.
	OnClose string `json:"onClose,omitempty"`

	// OpenAs: How to open a link. Not supported by Chat apps.
	//
	// Possible values:
	//   "FULL_SIZE" - The link opens as a full size window (if that's the
	// frame used by the client.
	//   "OVERLAY" - The link opens as an overlay, such as a pop-up.
	OpenAs string `json:"openAs,omitempty"`

	// Url: The URL to open.
	Url string `json:"url,omitempty"`

	// ForceSendFields is a list of field names (e.g. "OnClose") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "OnClose") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1OpenLink) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1OpenLink
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Section: A section contains a collection of widgets
// that are rendered vertically in the order that they are specified.
type GoogleAppsCardV1Section struct {
	// Collapsible: Indicates whether this section is collapsible.
	// Collapsible sections hide some or all widgets, but users can expand
	// the section to reveal the hidden widgets by clicking **Show more**.
	// Users can hide the widgets again by clicking **Show less**. To
	// determine which widgets are hidden, specify
	// `uncollapsibleWidgetsCount`.
	Collapsible bool `json:"collapsible,omitempty"`

	// Header: Text that appears at the top of a section. Supports simple
	// HTML formatted text
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	Header string `json:"header,omitempty"`

	// UncollapsibleWidgetsCount: The number of uncollapsible widgets which
	// remain visible even when a section is collapsed. For example, when a
	// section contains five widgets and the `uncollapsibleWidgetsCount` is
	// set to `2`, the first two widgets are always shown and the last three
	// are collapsed by default. The `uncollapsibleWidgetsCount` is taken
	// into account only when `collapsible` is `true`.
	UncollapsibleWidgetsCount int64 `json:"uncollapsibleWidgetsCount,omitempty"`

	// Widgets: All the widgets in the section. Must contain at least 1
	// widget.
	Widgets []*GoogleAppsCardV1Widget `json:"widgets,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Collapsible") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Collapsible") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Section) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Section
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SelectionInput: A widget that creates a UI item with
// options for users to select. For example, a dropdown menu or check
// list. Chat apps receive and can process the value of entered text
// during form input events. For details about working with form inputs,
// see Receive form data
// (https://developers.google.com/chat/how-tos/dialogs#receive_form_data_from_dialogs).
// When you need to collect data from users that matches options you
// set, use a selection input. To collect abstract data from users, use
// the text input widget instead.
type GoogleAppsCardV1SelectionInput struct {
	// Items: An array of the selected items. For example, all the selected
	// check boxes.
	Items []*GoogleAppsCardV1SelectionItem `json:"items,omitempty"`

	// Label: The text that appears above the selection input field in the
	// user interface. Specify text that helps the user enter the
	// information your app needs. For example, if users are selecting the
	// urgency of a work ticket from a drop-down menu, the label might be
	// "Urgency" or "Select urgency".
	Label string `json:"label,omitempty"`

	// Name: The name by which the selection input is identified in a form
	// input event. For details about working with form inputs, see Receive
	// form data
	// (https://developers.google.com/chat/how-tos/dialogs#receive_form_data_from_dialogs).
	Name string `json:"name,omitempty"`

	// OnChangeAction: If specified, the form is submitted when the
	// selection changes. If not specified, you must specify a separate
	// button that submits the form. For details about working with form
	// inputs, see Receive form data
	// (https://developers.google.com/chat/how-tos/dialogs#receive_form_data_from_dialogs).
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`

	// Type: The way that an option appears to users. Different options
	// support different types of interactions. For example, users can
	// enable multiple check boxes, but can only select one value from a
	// dropdown menu. Each selection input supports one type of selection.
	// Mixing check boxes and switches, for example, is not supported.
	//
	// Possible values:
	//   "CHECK_BOX" - A set of checkboxes. Users can select multiple check
	// boxes per selection input.
	//   "RADIO_BUTTON" - A set of radio buttons. Users can select one radio
	// button per selection input.
	//   "SWITCH" - A set of switches. Users can turn on multiple switches
	// at once per selection input.
	//   "DROPDOWN" - A dropdown menu. Users can select one dropdown menu
	// item per selection input.
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Items") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1SelectionInput) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SelectionInput
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SelectionItem: A selectable item in a selection
// input, such as a check box or a switch.
type GoogleAppsCardV1SelectionItem struct {
	// Selected: When `true`, more than one item is selected. If more than
	// one item is selected for radio buttons and dropdown menus, the first
	// selected item is received and the ones after are ignored.
	Selected bool `json:"selected,omitempty"`

	// Text: The text displayed to users.
	Text string `json:"text,omitempty"`

	// Value: The value associated with this item. The client should use
	// this as a form input value. For details about working with form
	// inputs, see Receive form data
	// (https://developers.google.com/chat/how-tos/dialogs#receive_form_data_from_dialogs).
	Value string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Selected") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Selected") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1SelectionItem) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SelectionItem
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SuggestionItem: One suggested value that users can
// enter in a text input field.
type GoogleAppsCardV1SuggestionItem struct {
	// Text: The value of a suggested input to a text input field. This is
	// equivalent to what users would enter themselves.
	Text string `json:"text,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Text") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Text") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1SuggestionItem) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SuggestionItem
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Suggestions: Suggested values that users can enter.
// These values appear when users click inside the text input field. As
// users type, the suggested values dynamically filter to match what the
// users have typed. For example, a text input field for programming
// language might suggest Java, JavaScript, Python, and C++. When users
// start typing "Jav", the list of suggestions filters to show just Java
// and JavaScript. Suggested values help guide users to enter values
// that your app can make sense of. When referring to JavaScript, some
// users might enter "javascript" and others "java script". Suggesting
// "JavaScript" can standardize how users interact with your app. When
// specified, `TextInput.type` is always `SINGLE_LINE`, even if it is
// set to `MULTIPLE_LINE`.
type GoogleAppsCardV1Suggestions struct {
	// Items: A list of suggestions used for autocomplete recommendations in
	// text input fields.
	Items []*GoogleAppsCardV1SuggestionItem `json:"items,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Items") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Suggestions) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Suggestions
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1SwitchControl: Either a toggle-style switch or a
// checkbox inside a `decoratedText` widget. Only supported on the
// `decoratedText` widget.
type GoogleAppsCardV1SwitchControl struct {
	// ControlType: How the switch appears in the user interface.
	//
	// Possible values:
	//   "SWITCH" - A toggle-style switch.
	//   "CHECKBOX" - Deprecated in favor of `CHECK_BOX`.
	//   "CHECK_BOX" - A checkbox.
	ControlType string `json:"controlType,omitempty"`

	// Name: The name by which the switch widget is identified in a form
	// input event. For details about working with form inputs, see Receive
	// form data
	// (https://developers.google.com/chat/how-tos/dialogs#receive_form_data_from_dialogs).
	Name string `json:"name,omitempty"`

	// OnChangeAction: The action to perform when the switch state is
	// changed, such as what function to run.
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`

	// Selected: When `true`, the switch is selected.
	Selected bool `json:"selected,omitempty"`

	// Value: The value entered by a user, returned as part of a form input
	// event. For details about working with form inputs, see Receive form
	// data
	// (https://developers.google.com/chat/how-tos/dialogs#receive_form_data_from_dialogs).
	Value string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ControlType") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ControlType") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1SwitchControl) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1SwitchControl
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1TextInput: A field in which users can enter text.
// Supports suggestions and on-change actions. Chat apps receive and can
// process the value of entered text during form input events. For
// details about working with form inputs, see Receive form data
// (https://developers.google.com/chat/how-tos/dialogs#receive_form_data_from_dialogs).
// When you need to collect abstract data from users, use a text input.
// To collect defined data from users, use the selection input widget
// instead.
type GoogleAppsCardV1TextInput struct {
	// AutoCompleteAction: Optional. Specify what action to take when the
	// text input field provides suggestions to users who interact with it.
	// If unspecified, the suggestions are set by `initialSuggestions` and
	// are processed by the client. If specified, the app takes the action
	// specified here, such as running a custom function. Supported by
	// Google Workspace Add-ons, but not Chat apps. Support by Chat apps
	// coming soon.
	AutoCompleteAction *GoogleAppsCardV1Action `json:"autoCompleteAction,omitempty"`

	// HintText: Text that appears below the text input field meant to
	// assist users by prompting them to enter a certain value. This text is
	// always visible. Required if `label` is unspecified. Otherwise,
	// optional.
	HintText string `json:"hintText,omitempty"`

	// InitialSuggestions: Suggested values that users can enter. These
	// values appear when users click inside the text input field. As users
	// type, the suggested values dynamically filter to match what the users
	// have typed. For example, a text input field for programming language
	// might suggest Java, JavaScript, Python, and C++. When users start
	// typing "Jav", the list of suggestions filters to show just Java and
	// JavaScript. Suggested values help guide users to enter values that
	// your app can make sense of. When referring to JavaScript, some users
	// might enter "javascript" and others "java script". Suggesting
	// "JavaScript" can standardize how users interact with your app. When
	// specified, `TextInput.type` is always `SINGLE_LINE`, even if it is
	// set to `MULTIPLE_LINE`.
	InitialSuggestions *GoogleAppsCardV1Suggestions `json:"initialSuggestions,omitempty"`

	// Label: The text that appears above the text input field in the user
	// interface. Specify text that helps the user enter the information
	// your app needs. For example, if you are asking someone's name, but
	// specifically need their surname, write "surname" instead of "name".
	// Required if `hintText` is unspecified. Otherwise, optional.
	Label string `json:"label,omitempty"`

	// Name: The name by which the text input is identified in a form input
	// event. For details about working with form inputs, see Receive form
	// data
	// (https://developers.google.com/chat/how-tos/dialogs#receive_form_data_from_dialogs).
	Name string `json:"name,omitempty"`

	// OnChangeAction: What to do when a change occurs in the text input
	// field. Examples of changes include a user adding to the field, or
	// deleting text. Examples of actions to take include running a custom
	// function or opening a dialog
	// (https://developers.google.com/chat/how-tos/dialogs) in Google Chat.
	OnChangeAction *GoogleAppsCardV1Action `json:"onChangeAction,omitempty"`

	// Type: How a text input field appears in the user interface. For
	// example, whether the field is single or multi-line.
	//
	// Possible values:
	//   "SINGLE_LINE" - The text input field has a fixed height of one
	// line.
	//   "MULTIPLE_LINE" - The text input field has a fixed height of
	// multiple lines.
	Type string `json:"type,omitempty"`

	// Value: The value entered by a user, returned as part of a form input
	// event. For details about working with form inputs, see Receive form
	// data
	// (https://developers.google.com/chat/how-tos/dialogs#receive_form_data_from_dialogs).
	Value string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AutoCompleteAction")
	// to unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AutoCompleteAction") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1TextInput) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1TextInput
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1TextParagraph: A paragraph of text that supports
// formatting. See Text formatting
// (https://developers.google.com/workspace/add-ons/concepts/widgets#text_formatting)
// for details.
type GoogleAppsCardV1TextParagraph struct {
	// Text: The text that's shown in the widget.
	Text string `json:"text,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Text") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Text") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1TextParagraph) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1TextParagraph
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GoogleAppsCardV1Widget: Each card is made up of widgets. A widget is
// a composite object that can represent one of text, images, buttons,
// and other object types.
type GoogleAppsCardV1Widget struct {
	// ButtonList: A list of buttons. For example, the following JSON
	// creates two buttons. The first is a blue text button and the second
	// is an image button that opens a link: ``` "buttonList": { "buttons":
	// [ { "text": "Edit", "color": { "red": 0, "green": 0, "blue": 1,
	// "alpha": 1 }, "disabled": true, }, { "icon": { "knownIcon": "INVITE",
	// "altText": "check calendar" }, "onClick": { "openLink": { "url":
	// "https://example.com/calendar" } } } ] } ```
	ButtonList *GoogleAppsCardV1ButtonList `json:"buttonList,omitempty"`

	// DateTimePicker: Displays a selection/input widget for date, time, or
	// date and time. Not supported by Chat apps. Support by Chat apps is
	// coming soon. For example, the following JSON creates a datetime
	// picker to schedule an appointment: ``` "dateTimePicker": { "name":
	// "appointment_time", "label": "Book your appointment at:", "type":
	// "DATE_AND_TIME", "valueMsEpoch": "796435200000" } ```
	DateTimePicker *GoogleAppsCardV1DateTimePicker `json:"dateTimePicker,omitempty"`

	// DecoratedText: Displays a decorated text item. For example, the
	// following JSON creates a decorated text widget showing email address:
	// ``` "decoratedText": { "icon": { "knownIcon": "EMAIL" }, "topLabel":
	// "Email Address", "text": "sasha@example.com", "bottomLabel": "This is
	// a new Email address!", "switchControl": { "name":
	// "has_send_welcome_email_to_sasha", "selected": false, "controlType":
	// "CHECKBOX" } } ```
	DecoratedText *GoogleAppsCardV1DecoratedText `json:"decoratedText,omitempty"`

	// Divider: Displays a horizontal line divider between widgets. For
	// example, the following JSON creates a divider: ``` "divider": { } ```
	Divider *GoogleAppsCardV1Divider `json:"divider,omitempty"`

	// Grid: Displays a grid with a collection of items. A grid supports any
	// number of columns and items. The number of rows is determined by the
	// upper bounds of the number items divided by the number of columns. A
	// grid with 10 items and 2 columns has 5 rows. A grid with 11 items and
	// 2 columns has 6 rows. For example, the following JSON creates a 2
	// column grid with a single item: ``` "grid": { "title": "A fine
	// collection of items", "columnCount": 2, "borderStyle": { "type":
	// "STROKE", "cornerRadius": 4 }, "items": [ { "image": { "imageUri":
	// "https://www.example.com/image.png", "cropStyle": { "type": "SQUARE"
	// }, "borderStyle": { "type": "STROKE" } }, "title": "An item",
	// "textAlignment": "CENTER" } ], "onClick": { "openLink": { "url":
	// "https://www.example.com" } } } ```
	Grid *GoogleAppsCardV1Grid `json:"grid,omitempty"`

	// Image: Displays an image. For example, the following JSON creates an
	// image with alternative text: ``` "image": { "imageUrl":
	// "https://developers.google.com/chat/images/quickstart-app-avatar.png",
	//  "altText": "Chat app avatar" } ```
	Image *GoogleAppsCardV1Image `json:"image,omitempty"`

	// SelectionInput: Displays a selection control that lets users select
	// items. Selection controls can be check boxes, radio buttons,
	// switches, or dropdown menus. For example, the following JSON creates
	// a dropdown menu that lets users choose a size: ``` "selectionInput":
	// { "name": "size", "label": "Size" "type": "DROPDOWN", "items": [ {
	// "text": "S", "value": "small", "selected": false }, { "text": "M",
	// "value": "medium", "selected": true }, { "text": "L", "value":
	// "large", "selected": false }, { "text": "XL", "value": "extra_large",
	// "selected": false } ] } ```
	SelectionInput *GoogleAppsCardV1SelectionInput `json:"selectionInput,omitempty"`

	// TextInput: Displays a text box that users can type into. For example,
	// the following JSON creates a text input for an email address: ```
	// "textInput": { "name": "mailing_address", "label": "Mailing Address"
	// } ``` As another example, the following JSON creates a text input for
	// a programming language with static suggestions: ``` "textInput": {
	// "name": "preferred_programing_language", "label": "Preferred
	// Language", "initialSuggestions": { "items": [ { "text": "C++" }, {
	// "text": "Java" }, { "text": "JavaScript" }, { "text": "Python" } ] }
	// } ```
	TextInput *GoogleAppsCardV1TextInput `json:"textInput,omitempty"`

	// TextParagraph: Displays a text paragraph. Supports simple HTML
	// formatted text
	// (https://developers.google.com/apps-script/add-ons/concepts/widgets#text_formatting).
	// For example, the following JSON creates a bolded text: ```
	// "textParagraph": { "text": " *bold text*" } ```
	TextParagraph *GoogleAppsCardV1TextParagraph `json:"textParagraph,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ButtonList") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ButtonList") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GoogleAppsCardV1Widget) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleAppsCardV1Widget
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Image: An image that is specified by a URL and can have an onclick
// action.
type Image struct {
	// AspectRatio: The aspect ratio of this image (width/height). This
	// field allows clients to reserve the right height for the image while
	// waiting for it to load. It's not meant to override the native aspect
	// ratio of the image. If unset, the server fills it by prefetching the
	// image.
	AspectRatio float64 `json:"aspectRatio,omitempty"`

	// ImageUrl: The URL of the image.
	ImageUrl string `json:"imageUrl,omitempty"`

	// OnClick: The onclick action.
	OnClick *OnClick `json:"onClick,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AspectRatio") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AspectRatio") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Image) MarshalJSON() ([]byte, error) {
	type NoMethod Image
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

func (s *Image) UnmarshalJSON(data []byte) error {
	type NoMethod Image
	var s1 struct {
		AspectRatio gensupport.JSONFloat64 `json:"aspectRatio"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.AspectRatio = float64(s1.AspectRatio)
	return nil
}

// ImageButton: An image button with an onclick action.
type ImageButton struct {
	// Icon: The icon specified by an enum that indices to an icon provided
	// by Chat API.
	//
	// Possible values:
	//   "ICON_UNSPECIFIED"
	//   "AIRPLANE"
	//   "BOOKMARK"
	//   "BUS"
	//   "CAR"
	//   "CLOCK"
	//   "CONFIRMATION_NUMBER_ICON"
	//   "DOLLAR"
	//   "DESCRIPTION"
	//   "EMAIL"
	//   "EVENT_PERFORMER"
	//   "EVENT_SEAT"
	//   "FLIGHT_ARRIVAL"
	//   "FLIGHT_DEPARTURE"
	//   "HOTEL"
	//   "HOTEL_ROOM_TYPE"
	//   "INVITE"
	//   "MAP_PIN"
	//   "MEMBERSHIP"
	//   "MULTIPLE_PEOPLE"
	//   "OFFER"
	//   "PERSON"
	//   "PHONE"
	//   "RESTAURANT_ICON"
	//   "SHOPPING_CART"
	//   "STAR"
	//   "STORE"
	//   "TICKET"
	//   "TRAIN"
	//   "VIDEO_CAMERA"
	//   "VIDEO_PLAY"
	Icon string `json:"icon,omitempty"`

	// IconUrl: The icon specified by a URL.
	IconUrl string `json:"iconUrl,omitempty"`

	// Name: The name of this image_button which will be used for
	// accessibility. Default value will be provided if developers don't
	// specify.
	Name string `json:"name,omitempty"`

	// OnClick: The onclick action.
	OnClick *OnClick `json:"onClick,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Icon") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Icon") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ImageButton) MarshalJSON() ([]byte, error) {
	type NoMethod ImageButton
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Inputs: Types of data inputs for widgets. Users enter data with these
// inputs.
type Inputs struct {
	// DateInput: Date input values. Not supported by Chat apps.
	DateInput *DateInput `json:"dateInput,omitempty"`

	// DateTimeInput: Date and time input values. Not supported by Chat
	// apps.
	DateTimeInput *DateTimeInput `json:"dateTimeInput,omitempty"`

	// StringInputs: Input parameter for regular widgets. For single-valued
	// widgets, it is a single value list. For multi-valued widgets, such as
	// checkbox, all the values are presented.
	StringInputs *StringInputs `json:"stringInputs,omitempty"`

	// TimeInput: Time input values. Not supported by Chat apps.
	TimeInput *TimeInput `json:"timeInput,omitempty"`

	// ForceSendFields is a list of field names (e.g. "DateInput") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DateInput") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Inputs) MarshalJSON() ([]byte, error) {
	type NoMethod Inputs
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// KeyValue: A UI element contains a key (label) and a value (content).
// And this element may also contain some actions such as onclick
// button.
type KeyValue struct {
	// BottomLabel: The text of the bottom label. Formatted text supported.
	BottomLabel string `json:"bottomLabel,omitempty"`

	// Button: A button that can be clicked to trigger an action.
	Button *Button `json:"button,omitempty"`

	// Content: The text of the content. Formatted text supported and always
	// required.
	Content string `json:"content,omitempty"`

	// ContentMultiline: If the content should be multiline.
	ContentMultiline bool `json:"contentMultiline,omitempty"`

	// Icon: An enum value that will be replaced by the Chat API with the
	// corresponding icon image.
	//
	// Possible values:
	//   "ICON_UNSPECIFIED"
	//   "AIRPLANE"
	//   "BOOKMARK"
	//   "BUS"
	//   "CAR"
	//   "CLOCK"
	//   "CONFIRMATION_NUMBER_ICON"
	//   "DOLLAR"
	//   "DESCRIPTION"
	//   "EMAIL"
	//   "EVENT_PERFORMER"
	//   "EVENT_SEAT"
	//   "FLIGHT_ARRIVAL"
	//   "FLIGHT_DEPARTURE"
	//   "HOTEL"
	//   "HOTEL_ROOM_TYPE"
	//   "INVITE"
	//   "MAP_PIN"
	//   "MEMBERSHIP"
	//   "MULTIPLE_PEOPLE"
	//   "OFFER"
	//   "PERSON"
	//   "PHONE"
	//   "RESTAURANT_ICON"
	//   "SHOPPING_CART"
	//   "STAR"
	//   "STORE"
	//   "TICKET"
	//   "TRAIN"
	//   "VIDEO_CAMERA"
	//   "VIDEO_PLAY"
	Icon string `json:"icon,omitempty"`

	// IconUrl: The icon specified by a URL.
	IconUrl string `json:"iconUrl,omitempty"`

	// OnClick: The onclick action. Only the top label, bottom label and
	// content region are clickable.
	OnClick *OnClick `json:"onClick,omitempty"`

	// TopLabel: The text of the top label. Formatted text supported.
	TopLabel string `json:"topLabel,omitempty"`

	// ForceSendFields is a list of field names (e.g. "BottomLabel") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "BottomLabel") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *KeyValue) MarshalJSON() ([]byte, error) {
	type NoMethod KeyValue
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type ListMembershipsResponse struct {
	// Memberships: List of memberships in the requested (or first) page.
	Memberships []*Membership `json:"memberships,omitempty"`

	// NextPageToken: A token that can be sent as `pageToken` to retrieve
	// the next page of results. If empty, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Memberships") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Memberships") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ListMembershipsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListMembershipsResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type ListSpacesResponse struct {
	// NextPageToken: A token that can be sent as `pageToken` to retrieve
	// the next page of results. If empty, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// Spaces: List of spaces in the requested (or first) page.
	Spaces []*Space `json:"spaces,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "NextPageToken") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ListSpacesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListSpacesResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// MatchedUrl: A matched url in a Chat message. Chat apps can preview
// matched URLs. For more information, refer to Preview links
// (https://developers.google.com/chat/how-tos/preview-links).
type MatchedUrl struct {
	// Url: Output only. The url that was matched.
	Url string `json:"url,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Url") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Url") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *MatchedUrl) MarshalJSON() ([]byte, error) {
	type NoMethod MatchedUrl
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Media: Media resource.
type Media struct {
	// ResourceName: Name of the media resource.
	ResourceName string `json:"resourceName,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "ResourceName") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ResourceName") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Media) MarshalJSON() ([]byte, error) {
	type NoMethod Media
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Membership: Represents a membership relation in Google Chat, such as
// whether a user or Chat app is invited to, part of, or absent from a
// space.
type Membership struct {
	// CreateTime: Output only. The creation time of the membership, such as
	// when a member joined or was invited to join a space.
	CreateTime string `json:"createTime,omitempty"`

	// Member: The Google Chat user or app the membership corresponds to.
	Member *User `json:"member,omitempty"`

	// Name: Resource name of the membership, assigned by the server.
	// Format: spaces/{space}/members/{member}
	Name string `json:"name,omitempty"`

	// Role: Output only. User's role within a Chat space, which determines
	// their permitted actions in the space.
	//
	// Possible values:
	//   "MEMBERSHIP_ROLE_UNSPECIFIED" - Default value. The user isn't a
	// member of the space, but might be invited.
	//   "ROLE_MEMBER" - A member of the space. The user has basic
	// permissions, like sending messages to the space. In 1:1 and unnamed
	// group conversations, everyone has this role.
	//   "ROLE_MANAGER" - A space manager. The user has all basic
	// permissions plus administrative permissions that allow them to manage
	// the space, like adding or removing members. Only supported in
	// SpaceType.SPACE.
	Role string `json:"role,omitempty"`

	// State: Output only. State of the membership.
	//
	// Possible values:
	//   "MEMBERSHIP_STATE_UNSPECIFIED" - Default, do not use.
	//   "JOINED" - The user has joined the space.
	//   "INVITED" - The user has been invited, is able to join the space,
	// but currently has not joined.
	//   "NOT_A_MEMBER" - The user is not a member of the space, has not
	// been invited and is not able to join the space.
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CreateTime") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Membership) MarshalJSON() ([]byte, error) {
	type NoMethod Membership
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Message: A message in Google Chat.
type Message struct {
	// ActionResponse: Input only. Parameters that a Chat app can use to
	// configure how its response is posted.
	ActionResponse *ActionResponse `json:"actionResponse,omitempty"`

	// Annotations: Output only. Annotations associated with the text in
	// this message.
	Annotations []*Annotation `json:"annotations,omitempty"`

	// ArgumentText: Output only. Plain-text body of the message with all
	// Chat app mentions stripped out.
	ArgumentText string `json:"argumentText,omitempty"`

	// Attachment: User-uploaded attachment.
	Attachment []*Attachment `json:"attachment,omitempty"`

	// Cards: Deprecated: Use `cards_v2` instead. Rich, formatted and
	// interactive cards that can be used to display UI elements such as:
	// formatted texts, buttons, clickable images. Cards are normally
	// displayed below the plain-text body of the message. `cards` and
	// `cards_v2` can have a maximum size of 32 KB.
	Cards []*Card `json:"cards,omitempty"`

	// CardsV2: Richly formatted and interactive cards that display UI
	// elements and editable widgets, such as: - Formatted text - Buttons -
	// Clickable images - Checkboxes - Radio buttons - Input widgets. Cards
	// are usually displayed below the text body of a Chat message, but can
	// situationally appear other places, such as dialogs
	// (https://developers.google.com/chat/how-tos/dialogs). Each card can
	// have a maximum size of 32 KB. The `cardId` is a unique identifier
	// among cards in the same message and for identifying user input
	// values. Currently supported widgets include: - `TextParagraph` -
	// `DecoratedText` - `Image` - `ButtonList` - `Divider` - `TextInput` -
	// `SelectionInput` - `Grid`
	CardsV2 []*CardWithId `json:"cardsV2,omitempty"`

	// ClientAssignedMessageId: A custom name for a Chat message assigned at
	// creation. Must start with `client-` and contain only lowercase
	// letters, numbers, and hyphens up to 63 characters in length. Specify
	// this field to get, update, or delete the message with the specified
	// value. For example usage, see Name a created message
	// (https://developers.google.com/chat/api/guides/crudl/messages#name_a_created_message).
	ClientAssignedMessageId string `json:"clientAssignedMessageId,omitempty"`

	// CreateTime: Output only. The time at which the message was created in
	// Google Chat server.
	CreateTime string `json:"createTime,omitempty"`

	// FallbackText: A plain-text description of the message's cards, used
	// when the actual cards cannot be displayed (e.g. mobile
	// notifications).
	FallbackText string `json:"fallbackText,omitempty"`

	// LastUpdateTime: Output only. The time at which the message was last
	// edited by a user. If the message has never been edited, this field is
	// empty.
	LastUpdateTime string `json:"lastUpdateTime,omitempty"`

	// MatchedUrl: Output only. A URL in `spaces.messages.text` that matches
	// a link preview pattern. For more information, refer to Preview links
	// (https://developers.google.com/chat/how-tos/preview-links).
	MatchedUrl *MatchedUrl `json:"matchedUrl,omitempty"`

	// Name: Resource name in the form `spaces/*/messages/*`. Example:
	// `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`
	Name string `json:"name,omitempty"`

	// Sender: Output only. The user who created the message. If your Chat
	// app authenticates as a user
	// (https://developers.google.com/chat/api/guides/auth/users), the
	// output populates the user
	// (https://developers.google.com/chat/api/reference/rest/v1/User)
	// `name` and `type`.
	Sender *User `json:"sender,omitempty"`

	// SlashCommand: Output only. Slash command information, if applicable.
	SlashCommand *SlashCommand `json:"slashCommand,omitempty"`

	// Space: If your Chat app authenticates as a user
	// (https://developers.google.com/chat/api/guides/auth/users), the
	// output populates the space
	// (https://developers.google.com/chat/api/reference/rest/v1/spaces)
	// `name`.
	Space *Space `json:"space,omitempty"`

	// Text: Plain-text body of the message. The first link to an image,
	// video, web page, or other preview-able item generates a preview chip.
	Text string `json:"text,omitempty"`

	// Thread: The thread the message belongs to. For example usage, see
	// Start or reply to a message thread
	// (https://developers.google.com/chat/api/guides/crudl/messages#start_or_reply_to_a_message_thread).
	Thread *Thread `json:"thread,omitempty"`

	// ThreadReply: Output only. When `true`, the message is a response in a
	// reply thread. When `false`, the message is visible in the space's
	// top-level conversation as either the first message of a thread or a
	// message with no threaded replies. If the space doesn't support reply
	// in threads, this field is always `false`.
	ThreadReply bool `json:"threadReply,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "ActionResponse") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ActionResponse") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *Message) MarshalJSON() ([]byte, error) {
	type NoMethod Message
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// OnClick: An onclick action (e.g. open a link).
type OnClick struct {
	// Action: A form action will be triggered by this onclick if specified.
	Action *FormAction `json:"action,omitempty"`

	// OpenLink: This onclick triggers an open link action if specified.
	OpenLink *OpenLink `json:"openLink,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Action") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Action") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *OnClick) MarshalJSON() ([]byte, error) {
	type NoMethod OnClick
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// OpenLink: A link that opens a new window.
type OpenLink struct {
	// Url: The URL to open.
	Url string `json:"url,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Url") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Url") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *OpenLink) MarshalJSON() ([]byte, error) {
	type NoMethod OpenLink
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Section: A section contains a collection of widgets that are rendered
// (vertically) in the order that they are specified. Across all
// platforms, cards have a narrow fixed width, so there is currently no
// need for layout properties (e.g. float).
type Section struct {
	// Header: The header of the section, text formatted supported.
	Header string `json:"header,omitempty"`

	// Widgets: A section must contain at least 1 widget.
	Widgets []*WidgetMarkup `json:"widgets,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Header") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Header") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Section) MarshalJSON() ([]byte, error) {
	type NoMethod Section
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// SlashCommand: A slash command
// (https://developers.google.com/chat/how-tos/slash-commands) in Google
// Chat.
type SlashCommand struct {
	// CommandId: The id of the slash command invoked.
	CommandId int64 `json:"commandId,omitempty,string"`

	// ForceSendFields is a list of field names (e.g. "CommandId") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CommandId") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *SlashCommand) MarshalJSON() ([]byte, error) {
	type NoMethod SlashCommand
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// SlashCommandMetadata: Annotation metadata for slash commands (/).
type SlashCommandMetadata struct {
	// Bot: The Chat app whose command was invoked.
	Bot *User `json:"bot,omitempty"`

	// CommandId: The command id of the invoked slash command.
	CommandId int64 `json:"commandId,omitempty,string"`

	// CommandName: The name of the invoked slash command.
	CommandName string `json:"commandName,omitempty"`

	// TriggersDialog: Indicating whether the slash command is for a dialog.
	TriggersDialog bool `json:"triggersDialog,omitempty"`

	// Type: The type of slash command.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for the enum. DO NOT USE.
	//   "ADD" - Add Chat app to space.
	//   "INVOKE" - Invoke slash command in space.
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Bot") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Bot") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *SlashCommandMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod SlashCommandMetadata
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Space: A space in Google Chat. Spaces are conversations between two
// or more users or 1:1 messages between a user and a Chat app.
type Space struct {
	// AdminInstalled: Output only. Whether the Chat app was installed by a
	// Google Workspace administrator. Administrators can install a Chat app
	// for their domain, organizational unit, or a group of users.
	// Administrators can only install Chat apps for direct messaging
	// between users and the app. To support admin install, your app must
	// feature direct messaging.
	AdminInstalled bool `json:"adminInstalled,omitempty"`

	// DisplayName: The space's display name. Required when creating a space
	// (https://developers.google.com/chat/api/reference/rest/v1/spaces/create).
	// For direct messages, this field may be empty. Supports up to 128
	// characters.
	DisplayName string `json:"displayName,omitempty"`

	// Name: Resource name of the space. Format: spaces/{space}
	Name string `json:"name,omitempty"`

	// SingleUserBotDm: Optional. Whether the space is a DM between a Chat
	// app and a single human.
	SingleUserBotDm bool `json:"singleUserBotDm,omitempty"`

	// SpaceDetails: Details about the space including description and
	// rules.
	SpaceDetails *SpaceDetails `json:"spaceDetails,omitempty"`

	// SpaceThreadingState: Output only. The threading state in the Chat
	// space.
	//
	// Possible values:
	//   "SPACE_THREADING_STATE_UNSPECIFIED" - Reserved.
	//   "THREADED_MESSAGES" - Named spaces that support message threads.
	// When users respond to a message, they can reply in-thread, which
	// keeps their response in the context of the original message.
	//   "GROUPED_MESSAGES" - Named spaces where the conversation is
	// organized by topic. Topics and their replies are grouped together.
	//   "UNTHREADED_MESSAGES" - Direct messages (DMs) between two people
	// and group conversations between 3 or more people.
	SpaceThreadingState string `json:"spaceThreadingState,omitempty"`

	// Threaded: Output only. Deprecated: Use `spaceThreadingState` instead.
	// Whether messages are threaded in this space.
	Threaded bool `json:"threaded,omitempty"`

	// Type: Output only. Deprecated: Use `singleUserBotDm` or `spaceType`
	// (developer preview) instead. The type of a space.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED"
	//   "ROOM" - Conversations between two or more humans.
	//   "DM" - 1:1 Direct Message between a human and a Chat app, where all
	// messages are flat. Note that this does not include direct messages
	// between two humans.
	Type string `json:"type,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "AdminInstalled") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AdminInstalled") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *Space) MarshalJSON() ([]byte, error) {
	type NoMethod Space
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// SpaceDetails: Details about the space including description and
// rules.
type SpaceDetails struct {
	// Description: Optional. A description of the space. It could describe
	// the space's discussion topic, functional purpose, or participants.
	// Supports up to 150 characters.
	Description string `json:"description,omitempty"`

	// Guidelines: Optional. The space's rules, expectations, and etiquette.
	// Supports up to 5,000 characters.
	Guidelines string `json:"guidelines,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Description") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Description") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *SpaceDetails) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceDetails
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Status: The `Status` type defines a logical error model that is
// suitable for different programming environments, including REST APIs
// and RPC APIs. It is used by gRPC (https://github.com/grpc). Each
// `Status` message contains three pieces of data: error code, error
// message, and error details. You can find out more about this error
// model and how to work with it in the API Design Guide
// (https://cloud.google.com/apis/design/errors).
type Status struct {
	// Code: The status code, which should be an enum value of
	// google.rpc.Code.
	Code int64 `json:"code,omitempty"`

	// Details: A list of messages that carry the error details. There is a
	// common set of message types for APIs to use.
	Details []googleapi.RawMessage `json:"details,omitempty"`

	// Message: A developer-facing error message, which should be in
	// English. Any user-facing error message should be localized and sent
	// in the google.rpc.Status.details field, or localized by the client.
	Message string `json:"message,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Code") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Code") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Status) MarshalJSON() ([]byte, error) {
	type NoMethod Status
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// StringInputs: Input parameter for regular widgets. For single-valued
// widgets, it is a single value list. For multi-valued widgets, such as
// checkbox, all the values are presented.
type StringInputs struct {
	// Value: An array of strings entered by the user.
	Value []string `json:"value,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Value") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Value") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *StringInputs) MarshalJSON() ([]byte, error) {
	type NoMethod StringInputs
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// TextButton: A button with text and onclick action.
type TextButton struct {
	// OnClick: The onclick action of the button.
	OnClick *OnClick `json:"onClick,omitempty"`

	// Text: The text of the button.
	Text string `json:"text,omitempty"`

	// ForceSendFields is a list of field names (e.g. "OnClick") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "OnClick") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *TextButton) MarshalJSON() ([]byte, error) {
	type NoMethod TextButton
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// TextParagraph: A paragraph of text. Formatted text supported.
type TextParagraph struct {
	Text string `json:"text,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Text") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Text") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *TextParagraph) MarshalJSON() ([]byte, error) {
	type NoMethod TextParagraph
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Thread: A thread in Google Chat.
type Thread struct {
	// Name: Resource name of the thread. Example:
	// spaces/{space}/threads/{thread}
	Name string `json:"name,omitempty"`

	// ThreadKey: Optional. Opaque thread identifier. To start or add to a
	// thread, create a message and specify a `threadKey` or the
	// thread.name. For example usage, see Start or reply to a message
	// thread
	// (https://developers.google.com/chat/api/guides/crudl/messages#start_or_reply_to_a_message_thread).
	// For other requests, this is an output only field.
	ThreadKey string `json:"threadKey,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Name") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Name") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Thread) MarshalJSON() ([]byte, error) {
	type NoMethod Thread
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// TimeInput: Time input values. Not supported by Chat apps.
type TimeInput struct {
	// Hours: The hour on a 24-hour clock.
	Hours int64 `json:"hours,omitempty"`

	// Minutes: The number of minutes past the hour. Valid values are 0 to
	// 59.
	Minutes int64 `json:"minutes,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Hours") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Hours") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *TimeInput) MarshalJSON() ([]byte, error) {
	type NoMethod TimeInput
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// TimeZone: The timezone ID and offset from Coordinated Universal Time
// (UTC). Not supported by Chat apps.
type TimeZone struct {
	// Id: The IANA TZ (https://www.iana.org/time-zones) time zone database
	// code, such as "America/Toronto".
	Id string `json:"id,omitempty"`

	// Offset: The user timezone offset, in milliseconds, from Coordinated
	// Universal Time (UTC).
	Offset int64 `json:"offset,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Id") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Id") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *TimeZone) MarshalJSON() ([]byte, error) {
	type NoMethod TimeZone
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// User: A user in Google Chat.
type User struct {
	// DisplayName: Output only. The user's display name.
	DisplayName string `json:"displayName,omitempty"`

	// DomainId: Unique identifier of the user's Google Workspace domain.
	DomainId string `json:"domainId,omitempty"`

	// IsAnonymous: Output only. When `true`, the user is deleted or their
	// profile is not visible.
	IsAnonymous bool `json:"isAnonymous,omitempty"`

	// Name: Resource name for a Google Chat user. Format: `users/{user}`.
	// `users/app` can be used as an alias for the calling app bot user. For
	// human users, `{user}` is the same user identifier as: - the
	// `{person_id`} for the Person
	// (https://developers.google.com/people/api/rest/v1/people) in the
	// People API, where the Person `resource_name` is `people/{person_id}`.
	// For example, `users/123456789` in Chat API represents the same person
	// as `people/123456789` in People API. - the `id` for a user
	// (https://developers.google.com/admin-sdk/directory/reference/rest/v1/users)
	// in the Admin SDK Directory API.
	Name string `json:"name,omitempty"`

	// Type: User type.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for the enum. DO NOT USE.
	//   "HUMAN" - Human user.
	//   "BOT" - Chat app user.
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DisplayName") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *User) MarshalJSON() ([]byte, error) {
	type NoMethod User
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// UserMentionMetadata: Annotation metadata for user mentions (@).
type UserMentionMetadata struct {
	// Type: The type of user mention.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for the enum. DO NOT USE.
	//   "ADD" - Add user to space.
	//   "MENTION" - Mention user in space.
	Type string `json:"type,omitempty"`

	// User: The user mentioned.
	User *User `json:"user,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Type") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Type") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *UserMentionMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod UserMentionMetadata
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// WidgetMarkup: A widget is a UI element that presents texts, images,
// etc.
type WidgetMarkup struct {
	// Buttons: A list of buttons. Buttons is also oneof data and only one
	// of these fields should be set.
	Buttons []*Button `json:"buttons,omitempty"`

	// Image: Display an image in this widget.
	Image *Image `json:"image,omitempty"`

	// KeyValue: Display a key value item in this widget.
	KeyValue *KeyValue `json:"keyValue,omitempty"`

	// TextParagraph: Display a text paragraph in this widget.
	TextParagraph *TextParagraph `json:"textParagraph,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Buttons") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Buttons") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *WidgetMarkup) MarshalJSON() ([]byte, error) {
	type NoMethod WidgetMarkup
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// method id "chat.media.download":

type MediaDownloadCall struct {
	s            *Service
	resourceName string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Download: Downloads media. Download is supported on the URI
// `/v1/media/{+name}?alt=media`.
//
//   - resourceName: Name of the media that is being downloaded. See
//     ReadRequest.resource_name.
func (r *MediaService) Download(resourceName string) *MediaDownloadCall {
	c := &MediaDownloadCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resourceName = resourceName
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *MediaDownloadCall) Fields(s ...googleapi.Field) *MediaDownloadCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *MediaDownloadCall) IfNoneMatch(entityTag string) *MediaDownloadCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do and Download
// methods. Any pending HTTP request will be aborted if the provided
// context is canceled.
func (c *MediaDownloadCall) Context(ctx context.Context) *MediaDownloadCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *MediaDownloadCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *MediaDownloadCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/media/{+resourceName}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resourceName": c.resourceName,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Download fetches the API endpoint's "media" value, instead of the normal
// API response value. If the returned error is nil, the Response is guaranteed to
// have a 2xx status code. Callers must close the Response.Body as usual.
func (c *MediaDownloadCall) Download(opts ...googleapi.CallOption) (*http.Response, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("media")
	if err != nil {
		return nil, err
	}
	if err := googleapi.CheckResponse(res); err != nil {
		res.Body.Close()
		return nil, gensupport.WrapError(err)
	}
	return res, nil
}

// Do executes the "chat.media.download" call.
// Exactly one of *Media or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Media.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *MediaDownloadCall) Do(opts ...googleapi.CallOption) (*Media, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Media{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Downloads media. Download is supported on the URI `/v1/media/{+name}?alt=media`.",
	//   "flatPath": "v1/media/{mediaId}",
	//   "httpMethod": "GET",
	//   "id": "chat.media.download",
	//   "parameterOrder": [
	//     "resourceName"
	//   ],
	//   "parameters": {
	//     "resourceName": {
	//       "description": "Name of the media that is being downloaded. See ReadRequest.resource_name.",
	//       "location": "path",
	//       "pattern": "^.*$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/media/{+resourceName}",
	//   "response": {
	//     "$ref": "Media"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.readonly"
	//   ],
	//   "supportsMediaDownload": true
	// }

}

// method id "chat.spaces.get":

type SpacesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns a space. Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
// Supports user authentication
// (https://developers.google.com/chat/api/guides/auth/users) as part of
// the Google Workspace Developer Preview Program
// (https://developers.google.com/workspace/preview), which grants early
// access to certain features. User authentication
// (https://developers.google.com/chat/api/guides/auth/users) requires
// the `chat.spaces` or `chat.spaces.readonly` authorization scope.
//
//   - name: Resource name of the space, in the form "spaces/*". Format:
//     spaces/{space}.
func (r *SpacesService) Get(name string) *SpacesGetCall {
	c := &SpacesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesGetCall) Fields(s ...googleapi.Field) *SpacesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesGetCall) IfNoneMatch(entityTag string) *SpacesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesGetCall) Context(ctx context.Context) *SpacesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.get" call.
// Exactly one of *Space or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesGetCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Returns a space. Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts). Supports [user authentication](https://developers.google.com/chat/api/guides/auth/users) as part of the [Google Workspace Developer Preview Program](https://developers.google.com/workspace/preview), which grants early access to certain features. [User authentication](https://developers.google.com/chat/api/guides/auth/users) requires the `chat.spaces` or `chat.spaces.readonly` authorization scope.",
	//   "flatPath": "v1/spaces/{spacesId}",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.get",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the space, in the form \"spaces/*\". Format: spaces/{space}",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Space"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.spaces",
	//     "https://www.googleapis.com/auth/chat.spaces.readonly"
	//   ]
	// }

}

// method id "chat.spaces.list":

type SpacesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists spaces the caller is a member of. Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
// Supports user authentication
// (https://developers.google.com/chat/api/guides/auth/users) as part of
// the Google Workspace Developer Preview Program
// (https://developers.google.com/workspace/preview), which grants early
// access to certain features. User authentication
// (https://developers.google.com/chat/api/guides/auth/users) requires
// the `chat.spaces` or `chat.spaces.readonly` authorization scope.
// Lists spaces visible to the caller or authenticated user. Group chats
// and DMs aren't listed until the first message is sent.
func (r *SpacesService) List() *SpacesListCall {
	c := &SpacesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number
// of spaces to return. The service may return fewer than this value. If
// unspecified, at most 100 spaces are returned. The maximum value is
// 1000; values above 1000 are coerced to 1000. Negative values return
// an `INVALID_ARGUMENT` error.
func (c *SpacesListCall) PageSize(pageSize int64) *SpacesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token,
// received from a previous list spaces call. Provide this to retrieve
// the subsequent page. When paginating, the filter value should match
// the call that provided the page token. Passing a different value may
// lead to unexpected results.
func (c *SpacesListCall) PageToken(pageToken string) *SpacesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesListCall) Fields(s ...googleapi.Field) *SpacesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesListCall) IfNoneMatch(entityTag string) *SpacesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesListCall) Context(ctx context.Context) *SpacesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/spaces")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.list" call.
// Exactly one of *ListSpacesResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ListSpacesResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *SpacesListCall) Do(opts ...googleapi.CallOption) (*ListSpacesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListSpacesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Lists spaces the caller is a member of. Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts). Supports [user authentication](https://developers.google.com/chat/api/guides/auth/users) as part of the [Google Workspace Developer Preview Program](https://developers.google.com/workspace/preview), which grants early access to certain features. [User authentication](https://developers.google.com/chat/api/guides/auth/users) requires the `chat.spaces` or `chat.spaces.readonly` authorization scope. Lists spaces visible to the caller or authenticated user. Group chats and DMs aren't listed until the first message is sent.",
	//   "flatPath": "v1/spaces",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.list",
	//   "parameterOrder": [],
	//   "parameters": {
	//     "pageSize": {
	//       "description": "Optional. The maximum number of spaces to return. The service may return fewer than this value. If unspecified, at most 100 spaces are returned. The maximum value is 1000; values above 1000 are coerced to 1000. Negative values return an `INVALID_ARGUMENT` error.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "Optional. A page token, received from a previous list spaces call. Provide this to retrieve the subsequent page. When paginating, the filter value should match the call that provided the page token. Passing a different value may lead to unexpected results.",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/spaces",
	//   "response": {
	//     "$ref": "ListSpacesResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.spaces",
	//     "https://www.googleapis.com/auth/chat.spaces.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesListCall) Pages(ctx context.Context, f func(*ListSpacesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

// method id "chat.spaces.members.get":

type SpacesMembersGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns a membership. Requires authentication
// (https://developers.google.com/chat/api/guides/auth/). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
// Supports user authentication
// (https://developers.google.com/chat/api/guides/auth/users) as part of
// the Google Workspace Developer Preview Program
// (https://developers.google.com/workspace/preview), which grants early
// access to certain features. User authentication
// (https://developers.google.com/chat/api/guides/auth/users) requires
// the `chat.memberships` or `chat.memberships.readonly` authorization
// scope.
//
//   - name: Resource name of the membership to retrieve. Format:
//     spaces/{space}/members/{member}.
func (r *SpacesMembersService) Get(name string) *SpacesMembersGetCall {
	c := &SpacesMembersGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMembersGetCall) Fields(s ...googleapi.Field) *SpacesMembersGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesMembersGetCall) IfNoneMatch(entityTag string) *SpacesMembersGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMembersGetCall) Context(ctx context.Context) *SpacesMembersGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMembersGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.get" call.
// Exactly one of *Membership or error will be non-nil. Any non-2xx
// status code is an error. Response headers are in either
// *Membership.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified
// to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SpacesMembersGetCall) Do(opts ...googleapi.CallOption) (*Membership, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Membership{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Returns a membership. Requires [authentication](https://developers.google.com/chat/api/guides/auth/). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts). Supports [user authentication](https://developers.google.com/chat/api/guides/auth/users) as part of the [Google Workspace Developer Preview Program](https://developers.google.com/workspace/preview), which grants early access to certain features. [User authentication](https://developers.google.com/chat/api/guides/auth/users) requires the `chat.memberships` or `chat.memberships.readonly` authorization scope.",
	//   "flatPath": "v1/spaces/{spacesId}/members/{membersId}",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.members.get",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the membership to retrieve. Format: spaces/{space}/members/{member}",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/members/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Membership"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.memberships",
	//     "https://www.googleapis.com/auth/chat.memberships.readonly"
	//   ]
	// }

}

// method id "chat.spaces.members.list":

type SpacesMembersListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists memberships in a space. Requires authentication
// (https://developers.google.com/chat/api/guides/auth/). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
// Supports user authentication
// (https://developers.google.com/chat/api/guides/auth/users) as part of
// the Google Workspace Developer Preview Program
// (https://developers.google.com/workspace/preview), which grants early
// access to certain features. User authentication
// (https://developers.google.com/chat/api/guides/auth/users) requires
// the `chat.memberships` or `chat.memberships.readonly` authorization
// scope.
//
//   - parent: The resource name of the space for which to fetch a
//     membership list. Format: spaces/{space}.
func (r *SpacesMembersService) List(parent string) *SpacesMembersListCall {
	c := &SpacesMembersListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number
// of memberships to return. The service may return fewer than this
// value. If unspecified, at most 100 memberships are returned. The
// maximum value is 1000; values above 1000 are coerced to 1000.
// Negative values return an INVALID_ARGUMENT error.
func (c *SpacesMembersListCall) PageSize(pageSize int64) *SpacesMembersListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token,
// received from a previous call to list memberships. Provide this to
// retrieve the subsequent page. When paginating, all other parameters
// provided should match the call that provided the page token. Passing
// different values to the other parameters may lead to unexpected
// results.
func (c *SpacesMembersListCall) PageToken(pageToken string) *SpacesMembersListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMembersListCall) Fields(s ...googleapi.Field) *SpacesMembersListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesMembersListCall) IfNoneMatch(entityTag string) *SpacesMembersListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMembersListCall) Context(ctx context.Context) *SpacesMembersListCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMembersListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMembersListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/members")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.members.list" call.
// Exactly one of *ListMembershipsResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ListMembershipsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *SpacesMembersListCall) Do(opts ...googleapi.CallOption) (*ListMembershipsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListMembershipsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Lists memberships in a space. Requires [authentication](https://developers.google.com/chat/api/guides/auth/). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts). Supports [user authentication](https://developers.google.com/chat/api/guides/auth/users) as part of the [Google Workspace Developer Preview Program](https://developers.google.com/workspace/preview), which grants early access to certain features. [User authentication](https://developers.google.com/chat/api/guides/auth/users) requires the `chat.memberships` or `chat.memberships.readonly` authorization scope.",
	//   "flatPath": "v1/spaces/{spacesId}/members",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.members.list",
	//   "parameterOrder": [
	//     "parent"
	//   ],
	//   "parameters": {
	//     "pageSize": {
	//       "description": "The maximum number of memberships to return. The service may return fewer than this value. If unspecified, at most 100 memberships are returned. The maximum value is 1000; values above 1000 are coerced to 1000. Negative values return an INVALID_ARGUMENT error.",
	//       "format": "int32",
	//       "location": "query",
	//       "type": "integer"
	//     },
	//     "pageToken": {
	//       "description": "A page token, received from a previous call to list memberships. Provide this to retrieve the subsequent page. When paginating, all other parameters provided should match the call that provided the page token. Passing different values to the other parameters may lead to unexpected results.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "parent": {
	//       "description": "Required. The resource name of the space for which to fetch a membership list. Format: spaces/{space}",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+parent}/members",
	//   "response": {
	//     "$ref": "ListMembershipsResponse"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.memberships",
	//     "https://www.googleapis.com/auth/chat.memberships.readonly"
	//   ]
	// }

}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SpacesMembersListCall) Pages(ctx context.Context, f func(*ListMembershipsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken")) // reset paging to original point
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

// method id "chat.spaces.messages.create":

type SpacesMessagesCreateCall struct {
	s          *Service
	parent     string
	message    *Message
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a message. For example usage, see Create a message
// (https://developers.google.com/chat/api/guides/crudl/messages#create_a_message).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
// Supports user authentication
// (https://developers.google.com/chat/api/guides/auth/users) as part of
// the Google Workspace Developer Preview Program
// (https://developers.google.com/workspace/preview), which grants early
// access to certain features. User authentication
// (https://developers.google.com/chat/api/guides/auth/users) requires
// the `chat.messages` or `chat.messages.create` authorization scope.
// Because Chat provides authentication for webhooks
// (https://developers.google.com/chat/how-tos/webhooks) as part of the
// URL that's generated when a webhook is registered, webhooks can
// create messages without a service account or user authentication.
//
//   - parent: The resource name of the space in which to create a
//     message. Format: spaces/{space}.
func (r *SpacesMessagesService) Create(parent string, message *Message) *SpacesMessagesCreateCall {
	c := &SpacesMessagesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.message = message
	return c
}

// MessageId sets the optional parameter "messageId": A custom name for
// a Chat message assigned at creation. Must start with `client-` and
// contain only lowercase letters, numbers, and hyphens up to 63
// characters in length. Specify this field to get, update, or delete
// the message with the specified value. For example usage, see Name a
// created message
// (https://developers.google.com/chat/api/guides/crudl/messages#name_a_created_message).
func (c *SpacesMessagesCreateCall) MessageId(messageId string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("messageId", messageId)
	return c
}

// MessageReplyOption sets the optional parameter "messageReplyOption":
// Specifies whether a message starts a thread or replies to one. Only
// supported in named spaces.
//
// Possible values:
//
//	"MESSAGE_REPLY_OPTION_UNSPECIFIED" - Default. Starts a thread.
//	"REPLY_MESSAGE_FALLBACK_TO_NEW_THREAD" - Creates the message as a
//
// reply to the thread specified by thread ID or thread_key. If it
// fails, the message starts a new thread instead.
//
//	"REPLY_MESSAGE_OR_FAIL" - Creates the message as a reply to the
//
// thread specified by thread ID or thread_key. If it fails, a NOT_FOUND
// error is returned instead.
func (c *SpacesMessagesCreateCall) MessageReplyOption(messageReplyOption string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("messageReplyOption", messageReplyOption)
	return c
}

// RequestId sets the optional parameter "requestId": A unique request
// ID for this message. Specifying an existing request ID returns the
// message created with that ID instead of creating a new message.
func (c *SpacesMessagesCreateCall) RequestId(requestId string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("requestId", requestId)
	return c
}

// ThreadKey sets the optional parameter "threadKey": Deprecated: Use
// thread.thread_key instead. Opaque thread identifier. To start or add
// to a thread, create a message and specify a `threadKey` or the
// thread.name. For example usage, see Start or reply to a message
// thread
// (https://developers.google.com/chat/api/guides/crudl/messages#start_or_reply_to_a_message_thread).
func (c *SpacesMessagesCreateCall) ThreadKey(threadKey string) *SpacesMessagesCreateCall {
	c.urlParams_.Set("threadKey", threadKey)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesCreateCall) Fields(s ...googleapi.Field) *SpacesMessagesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesCreateCall) Context(ctx context.Context) *SpacesMessagesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.message)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/messages")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.create" call.
// Exactly one of *Message or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesCreateCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Creates a message. For example usage, see [Create a message](https://developers.google.com/chat/api/guides/crudl/messages#create_a_message). Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts). Supports [user authentication](https://developers.google.com/chat/api/guides/auth/users) as part of the [Google Workspace Developer Preview Program](https://developers.google.com/workspace/preview), which grants early access to certain features. [User authentication](https://developers.google.com/chat/api/guides/auth/users) requires the `chat.messages` or `chat.messages.create` authorization scope. Because Chat provides authentication for [webhooks](https://developers.google.com/chat/how-tos/webhooks) as part of the URL that's generated when a webhook is registered, webhooks can create messages without a service account or user authentication.",
	//   "flatPath": "v1/spaces/{spacesId}/messages",
	//   "httpMethod": "POST",
	//   "id": "chat.spaces.messages.create",
	//   "parameterOrder": [
	//     "parent"
	//   ],
	//   "parameters": {
	//     "messageId": {
	//       "description": "Optional. A custom name for a Chat message assigned at creation. Must start with `client-` and contain only lowercase letters, numbers, and hyphens up to 63 characters in length. Specify this field to get, update, or delete the message with the specified value. For example usage, see [Name a created message](https://developers.google.com/chat/api/guides/crudl/messages#name_a_created_message).",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "messageReplyOption": {
	//       "description": "Optional. Specifies whether a message starts a thread or replies to one. Only supported in named spaces.",
	//       "enum": [
	//         "MESSAGE_REPLY_OPTION_UNSPECIFIED",
	//         "REPLY_MESSAGE_FALLBACK_TO_NEW_THREAD",
	//         "REPLY_MESSAGE_OR_FAIL"
	//       ],
	//       "enumDescriptions": [
	//         "Default. Starts a thread.",
	//         "Creates the message as a reply to the thread specified by thread ID or thread_key. If it fails, the message starts a new thread instead.",
	//         "Creates the message as a reply to the thread specified by thread ID or thread_key. If it fails, a NOT_FOUND error is returned instead."
	//       ],
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "parent": {
	//       "description": "Required. The resource name of the space in which to create a message. Format: spaces/{space}",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "requestId": {
	//       "description": "Optional. A unique request ID for this message. Specifying an existing request ID returns the message created with that ID instead of creating a new message.",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "threadKey": {
	//       "description": "Optional. Deprecated: Use thread.thread_key instead. Opaque thread identifier. To start or add to a thread, create a message and specify a `threadKey` or the thread.name. For example usage, see [Start or reply to a message thread](https://developers.google.com/chat/api/guides/crudl/messages#start_or_reply_to_a_message_thread).",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+parent}/messages",
	//   "request": {
	//     "$ref": "Message"
	//   },
	//   "response": {
	//     "$ref": "Message"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.create"
	//   ]
	// }

}

// method id "chat.spaces.messages.delete":

type SpacesMessagesDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a message. For example usage, see Delete a message
// (https://developers.google.com/chat/api/guides/crudl/messages#delete_a_message).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
// Supports user authentication
// (https://developers.google.com/chat/api/guides/auth/users) as part of
// the Google Workspace Developer Preview Program
// (https://developers.google.com/workspace/preview), which grants early
// access to certain features. User authentication
// (https://developers.google.com/chat/api/guides/auth/users) requires
// the `chat.messages` authorization scope.
//
//   - name: Resource name of the message to be deleted, in the form
//     "spaces/*/messages/*" Example:
//     spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB.
func (r *SpacesMessagesService) Delete(name string) *SpacesMessagesDeleteCall {
	c := &SpacesMessagesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesDeleteCall) Fields(s ...googleapi.Field) *SpacesMessagesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesDeleteCall) Context(ctx context.Context) *SpacesMessagesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.delete" call.
// Exactly one of *Empty or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Deletes a message. For example usage, see [Delete a message](https://developers.google.com/chat/api/guides/crudl/messages#delete_a_message). Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts). Supports [user authentication](https://developers.google.com/chat/api/guides/auth/users) as part of the [Google Workspace Developer Preview Program](https://developers.google.com/workspace/preview), which grants early access to certain features. [User authentication](https://developers.google.com/chat/api/guides/auth/users) requires the `chat.messages` authorization scope.",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}",
	//   "httpMethod": "DELETE",
	//   "id": "chat.spaces.messages.delete",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the message to be deleted, in the form \"spaces/*/messages/*\" Example: spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Empty"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.messages"
	//   ]
	// }

}

// method id "chat.spaces.messages.get":

type SpacesMessagesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns a message. For example usage, see Read a message
// (https://developers.google.com/chat/api/guides/crudl/messages#read_a_message).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth). Fully supports
// Service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
// Supports user authentication
// (https://developers.google.com/chat/api/guides/auth/users) as part of
// the Google Workspace Developer Preview Program
// (https://developers.google.com/workspace/preview), which grants early
// access to certain features. User authentication
// (https://developers.google.com/chat/api/guides/auth/users) requires
// the `chat.messages` or `chat.messages.readonly` authorization scope.
// Note: Might return a message from a blocked member or space.
//
//   - name: Resource name of the message to retrieve. Format:
//     spaces/{space}/messages/{message} If the message begins with
//     `client-`, then it has a custom name assigned by a Chat app that
//     created it with the Chat REST API. That Chat app (but not others)
//     can pass the custom name to get, update, or delete the message. To
//     learn more, see [create and name a message]
//     (https://developers.google.com/chat/api/guides/crudl/messages#name_a_created_message).
func (r *SpacesMessagesService) Get(name string) *SpacesMessagesGetCall {
	c := &SpacesMessagesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesGetCall) Fields(s ...googleapi.Field) *SpacesMessagesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesMessagesGetCall) IfNoneMatch(entityTag string) *SpacesMessagesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesGetCall) Context(ctx context.Context) *SpacesMessagesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.get" call.
// Exactly one of *Message or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesGetCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Returns a message. For example usage, see [Read a message](https://developers.google.com/chat/api/guides/crudl/messages#read_a_message). Requires [authentication](https://developers.google.com/chat/api/guides/auth). Fully supports [Service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts). Supports [user authentication](https://developers.google.com/chat/api/guides/auth/users) as part of the [Google Workspace Developer Preview Program](https://developers.google.com/workspace/preview), which grants early access to certain features. [User authentication](https://developers.google.com/chat/api/guides/auth/users) requires the `chat.messages` or `chat.messages.readonly` authorization scope. Note: Might return a message from a blocked member or space.",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.messages.get",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the message to retrieve. Format: spaces/{space}/messages/{message} If the message begins with `client-`, then it has a custom name assigned by a Chat app that created it with the Chat REST API. That Chat app (but not others) can pass the custom name to get, update, or delete the message. To learn more, see [create and name a message] (https://developers.google.com/chat/api/guides/crudl/messages#name_a_created_message).",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Message"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.messages",
	//     "https://www.googleapis.com/auth/chat.messages.readonly"
	//   ]
	// }

}

// method id "chat.spaces.messages.patch":

type SpacesMessagesPatchCall struct {
	s          *Service
	name       string
	message    *Message
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates a message. For example usage, see Update a message
// (https://developers.google.com/chat/api/guides/crudl/messages#update_a_message).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth/). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
// Supports user authentication
// (https://developers.google.com/chat/api/guides/auth/users) as part of
// the Google Workspace Developer Preview Program
// (https://developers.google.com/workspace/preview), which grants early
// access to certain features. User authentication
// (https://developers.google.com/chat/api/guides/auth/users) requires
// the `chat.messages` authorization scope.
//
//   - name: Resource name in the form `spaces/*/messages/*`. Example:
//     `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`.
func (r *SpacesMessagesService) Patch(name string, message *Message) *SpacesMessagesPatchCall {
	c := &SpacesMessagesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.message = message
	return c
}

// AllowMissing sets the optional parameter "allowMissing": If `true`
// and the message is not found, a new message is created and
// `updateMask` is ignored. The specified message ID must be
// client-assigned
// (https://developers.google.com/chat/api/guides/crudl/messages#name_a_created_message)
// or the request fails.
func (c *SpacesMessagesPatchCall) AllowMissing(allowMissing bool) *SpacesMessagesPatchCall {
	c.urlParams_.Set("allowMissing", fmt.Sprint(allowMissing))
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The
// field paths to update. Separate multiple values with commas.
// Currently supported field paths: - text - cards (Requires service
// account authentication (/chat/api/guides/auth/service-accounts).) -
// cards_v2
func (c *SpacesMessagesPatchCall) UpdateMask(updateMask string) *SpacesMessagesPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesPatchCall) Fields(s ...googleapi.Field) *SpacesMessagesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesPatchCall) Context(ctx context.Context) *SpacesMessagesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.message)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.patch" call.
// Exactly one of *Message or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesPatchCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Updates a message. For example usage, see [Update a message](https://developers.google.com/chat/api/guides/crudl/messages#update_a_message). Requires [authentication](https://developers.google.com/chat/api/guides/auth/). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts). Supports [user authentication](https://developers.google.com/chat/api/guides/auth/users) as part of the [Google Workspace Developer Preview Program](https://developers.google.com/workspace/preview), which grants early access to certain features. [User authentication](https://developers.google.com/chat/api/guides/auth/users) requires the `chat.messages` authorization scope.",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}",
	//   "httpMethod": "PATCH",
	//   "id": "chat.spaces.messages.patch",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "allowMissing": {
	//       "description": "Optional. If `true` and the message is not found, a new message is created and `updateMask` is ignored. The specified message ID must be [client-assigned](https://developers.google.com/chat/api/guides/crudl/messages#name_a_created_message) or the request fails.",
	//       "location": "query",
	//       "type": "boolean"
	//     },
	//     "name": {
	//       "description": "Resource name in the form `spaces/*/messages/*`. Example: `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "updateMask": {
	//       "description": "Required. The field paths to update. Separate multiple values with commas. Currently supported field paths: - text - cards (Requires [service account authentication](/chat/api/guides/auth/service-accounts).) - cards_v2 ",
	//       "format": "google-fieldmask",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "request": {
	//     "$ref": "Message"
	//   },
	//   "response": {
	//     "$ref": "Message"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.messages"
	//   ]
	// }

}

// method id "chat.spaces.messages.update":

type SpacesMessagesUpdateCall struct {
	s          *Service
	name       string
	message    *Message
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Update: Updates a message. For example usage, see Update a message
// (https://developers.google.com/chat/api/guides/crudl/messages#update_a_message).
// Requires authentication
// (https://developers.google.com/chat/api/guides/auth/). Fully supports
// service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
// Supports user authentication
// (https://developers.google.com/chat/api/guides/auth/users) as part of
// the Google Workspace Developer Preview Program
// (https://developers.google.com/workspace/preview), which grants early
// access to certain features. User authentication
// (https://developers.google.com/chat/api/guides/auth/users) requires
// the `chat.messages` authorization scope.
//
//   - name: Resource name in the form `spaces/*/messages/*`. Example:
//     `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`.
func (r *SpacesMessagesService) Update(name string, message *Message) *SpacesMessagesUpdateCall {
	c := &SpacesMessagesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.message = message
	return c
}

// AllowMissing sets the optional parameter "allowMissing": If `true`
// and the message is not found, a new message is created and
// `updateMask` is ignored. The specified message ID must be
// client-assigned
// (https://developers.google.com/chat/api/guides/crudl/messages#name_a_created_message)
// or the request fails.
func (c *SpacesMessagesUpdateCall) AllowMissing(allowMissing bool) *SpacesMessagesUpdateCall {
	c.urlParams_.Set("allowMissing", fmt.Sprint(allowMissing))
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. The
// field paths to update. Separate multiple values with commas.
// Currently supported field paths: - text - cards (Requires service
// account authentication (/chat/api/guides/auth/service-accounts).) -
// cards_v2
func (c *SpacesMessagesUpdateCall) UpdateMask(updateMask string) *SpacesMessagesUpdateCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesUpdateCall) Fields(s ...googleapi.Field) *SpacesMessagesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesUpdateCall) Context(ctx context.Context) *SpacesMessagesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.message)
	if err != nil {
		return nil, err
	}
	reqHeaders.Set("Content-Type", "application/json")
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.update" call.
// Exactly one of *Message or error will be non-nil. Any non-2xx status
// code is an error. Response headers are in either
// *Message.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified
// was returned.
func (c *SpacesMessagesUpdateCall) Do(opts ...googleapi.CallOption) (*Message, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Message{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Updates a message. For example usage, see [Update a message](https://developers.google.com/chat/api/guides/crudl/messages#update_a_message). Requires [authentication](https://developers.google.com/chat/api/guides/auth/). Fully supports [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts). Supports [user authentication](https://developers.google.com/chat/api/guides/auth/users) as part of the [Google Workspace Developer Preview Program](https://developers.google.com/workspace/preview), which grants early access to certain features. [User authentication](https://developers.google.com/chat/api/guides/auth/users) requires the `chat.messages` authorization scope.",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}",
	//   "httpMethod": "PUT",
	//   "id": "chat.spaces.messages.update",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "allowMissing": {
	//       "description": "Optional. If `true` and the message is not found, a new message is created and `updateMask` is ignored. The specified message ID must be [client-assigned](https://developers.google.com/chat/api/guides/crudl/messages#name_a_created_message) or the request fails.",
	//       "location": "query",
	//       "type": "boolean"
	//     },
	//     "name": {
	//       "description": "Resource name in the form `spaces/*/messages/*`. Example: `spaces/AAAAAAAAAAA/messages/BBBBBBBBBBB.BBBBBBBBBBB`",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "updateMask": {
	//       "description": "Required. The field paths to update. Separate multiple values with commas. Currently supported field paths: - text - cards (Requires [service account authentication](/chat/api/guides/auth/service-accounts).) - cards_v2 ",
	//       "format": "google-fieldmask",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "request": {
	//     "$ref": "Message"
	//   },
	//   "response": {
	//     "$ref": "Message"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot",
	//     "https://www.googleapis.com/auth/chat.messages"
	//   ]
	// }

}

// method id "chat.spaces.messages.attachments.get":

type SpacesMessagesAttachmentsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the metadata of a message attachment. The attachment data
// is fetched using the media API
// (https://developers.google.com/chat/api/reference/rest/v1/media/download).
// Requires service account authentication
// (https://developers.google.com/chat/api/guides/auth/service-accounts).
//
//   - name: Resource name of the attachment, in the form
//     "spaces/*/messages/*/attachments/*".
func (r *SpacesMessagesAttachmentsService) Get(name string) *SpacesMessagesAttachmentsGetCall {
	c := &SpacesMessagesAttachmentsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *SpacesMessagesAttachmentsGetCall) Fields(s ...googleapi.Field) *SpacesMessagesAttachmentsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *SpacesMessagesAttachmentsGetCall) IfNoneMatch(entityTag string) *SpacesMessagesAttachmentsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *SpacesMessagesAttachmentsGetCall) Context(ctx context.Context) *SpacesMessagesAttachmentsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *SpacesMessagesAttachmentsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesMessagesAttachmentsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chat.spaces.messages.attachments.get" call.
// Exactly one of *Attachment or error will be non-nil. Any non-2xx
// status code is an error. Response headers are in either
// *Attachment.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified
// to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SpacesMessagesAttachmentsGetCall) Do(opts ...googleapi.CallOption) (*Attachment, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Attachment{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Gets the metadata of a message attachment. The attachment data is fetched using the [media API](https://developers.google.com/chat/api/reference/rest/v1/media/download). Requires [service account authentication](https://developers.google.com/chat/api/guides/auth/service-accounts).",
	//   "flatPath": "v1/spaces/{spacesId}/messages/{messagesId}/attachments/{attachmentsId}",
	//   "httpMethod": "GET",
	//   "id": "chat.spaces.messages.attachments.get",
	//   "parameterOrder": [
	//     "name"
	//   ],
	//   "parameters": {
	//     "name": {
	//       "description": "Required. Resource name of the attachment, in the form \"spaces/*/messages/*/attachments/*\".",
	//       "location": "path",
	//       "pattern": "^spaces/[^/]+/messages/[^/]+/attachments/[^/]+$",
	//       "required": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "v1/{+name}",
	//   "response": {
	//     "$ref": "Attachment"
	//   },
	//   "scopes": [
	//     "https://www.googleapis.com/auth/chat.bot"
	//   ]
	// }

}
